/**
 ***********************************************************************************************************************
 * Copyright (c) 2020, China Mobile Communications Group Co.,Ltd.
 * COPYRIGHT (C) 2006 - 2018,RT - Thread Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with 
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on 
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the 
 * specific language governing permissions and limitations under the License.
 *
 * @file        syslog.c
 *
 * @brief       This file provides the interface adaptation of linux's syslog.
 *
 * @revision
 * Date         Author          Notes
 * 2018-09-07   armink          First version.
 * 2020-03-25   OneOS Team      First version.
 ***********************************************************************************************************************
 */

#include <oneos_config.h>
#include <os_types.h>
#include <os_stddef.h>
#include <os_hw.h>
#include <string.h>
#include <dlog.h>
#include <syslog.h>
#include "../log_internal.h"

#ifdef DLOG_OUTPUT_FLOAT
#include <stdio.h>
#endif

/*
 * Reference:
 * http://pubs.opengroup.org/onlinepubs/7908799/xsh/syslog.h.html
 * https://www.gnu.org/software/libc/manual/html_node/Submitting-Syslog-Messages.html
 * http://man7.org/linux/man-pages/man3/syslog.3.html
 */

#ifdef DLOG_USING_SYSLOG

#ifndef DLOG_SYSLOG_IDENT_MAX_LEN
#define DLOG_SYSLOG_IDENT_MAX_LEN       DLOG_FILTER_TAG_MAX_LEN
#endif

static char      gs_local_ident[DLOG_SYSLOG_IDENT_MAX_LEN + 1];
static int       gs_local_facility = LOG_USER;
static os_bool_t gs_is_open = OS_FALSE;

/**
 * open connection to syslog
 *
 * @param ident is an arbitrary identification string which future syslog invocations will prefix to each message.
 * @param option is not using on ulog.
 * @param facility is the default facility code for this connection.
 */


/**
 ***********************************************************************************************************************
 * @brief           Open connection to syslog.
 *
 * @param[in]       ident           An arbitrary identification string which future syslog invocations will prefix
 *                                  to each message.
 * @param[in]       option          Not using on dlog.
 * @param[in]       facility        The default facility code for this connection.
 *
 * @return          None.
 ***********************************************************************************************************************
 */
void openlog(const char *ident, int option, int facility)
{
    os_base_t level;

    if (gs_is_open)
    {
        return;
    }

    dlog_init();

    level = os_hw_interrupt_disable();

    memset(gs_local_ident, 0, sizeof(gs_local_ident));
    
    if (ident)
    {
        strncpy(gs_local_ident, ident, DLOG_SYSLOG_IDENT_MAX_LEN);
    }
    else
    {
        strncpy(gs_local_ident, "OneOS", DLOG_SYSLOG_IDENT_MAX_LEN);
    }

    if (facility)
    {
        gs_local_facility = facility;
    }
    else
    {
        /* Default facility is LOG_USER */
        gs_local_facility = LOG_USER;
    }
    
    gs_is_open = OS_TRUE;

    os_hw_interrupt_enable(level);

    return;
}

static void vsyslog(int priority, const char *format, va_list args)
{
    if (LOG_FAC(priority) == 0)
    {
        /* Using local facility */
        priority |= gs_local_facility;
    }

    dlog_voutput((os_uint16_t)priority, gs_local_ident, OS_TRUE, format, args);
}

/**
 ***********************************************************************************************************************
 * @brief           Generates a log message.
 *
 * @param[in]       priority        Log priority, can be generated by the macro LOG_MAKEPRI.
 * @param[in]       format          Log format, like printf().
 * @param[in]       ...             The argument.
 *
 * @return          None.
 ***********************************************************************************************************************
 */
void syslog(int priority, const char *format, ...)
{
    va_list args;

    if (!gs_is_open)
    {
        openlog(OS_NULL, 0, 0);
    }
    
    /* Args point to the first variable parameter */
    va_start(args, format);
    vsyslog(priority, format, args);
    va_end(args);

    return;
}

/**
 ***********************************************************************************************************************
 * @brief           Close the syslog
 *
 * @param           None.
 *
 * @return          None.
 ***********************************************************************************************************************
 */
void closelog(void)
{
    if (!gs_is_open)
    {
        return;
    }

    dlog_deinit();
    gs_is_open = OS_FALSE;

    return;
}
#endif /* DLOG_USING_SYSLOG */

