#include <board.h>
#include <drv_cfg.h>
#include <os_hw.h>

#include <os_memory.h>
#include <sensors/sensor.h>
#include <drv_log.h>
#include <string.h>
#include <stdlib.h>

#ifdef OS_USING_SHELL
#include <shell.h>
#endif

#define OS_AT24CXX_I2C_ADDR                 0xA0 >> 1

#define OS_AT24CXX_PAGE                     0x02

#define I2C_PAGESIZE                        16

static os_err_t write_regs(struct os_i2c_client *i2c, os_uint8_t reg, os_uint8_t *buf, os_uint8_t len)
{
    os_uint8_t device_addr = 0;

    OS_ASSERT(len <= 8);
    
    if(reg > 255)
    {
        device_addr = OS_AT24CXX_I2C_ADDR | 0x01;
    }
    else
    {
        device_addr = OS_AT24CXX_I2C_ADDR;
    }

    i2c->client_addr = device_addr;
    return  os_i2c_client_write(i2c, reg, 1, buf, len);
}

static os_err_t read_regs(struct os_i2c_client *i2c, os_uint16_t reg, os_uint8_t *buf, os_uint8_t len)
{
    os_uint8_t device_addr = 0;
    
    if(reg > 255)
    {
        device_addr = OS_AT24CXX_I2C_ADDR | 0x01;
    }
    else
    {
        device_addr = OS_AT24CXX_I2C_ADDR;
    }

    i2c->client_addr = device_addr;
    return  os_i2c_client_read(i2c, reg, 1, buf, len);
}


void __at24cxx_test(const char *i2c_bus)
{
    int i;
    os_uint8_t reg = 0x8; 
    os_uint8_t read_buff[8];
    os_uint8_t write_buff[8];
    
    struct os_i2c_client *at24cxx_i2c = os_calloc(1, sizeof(struct os_i2c_client));;

    at24cxx_i2c->bus = (struct os_i2c_bus_device *)os_device_find(i2c_bus);
    if (at24cxx_i2c->bus == NULL)
    {
        LOG_EXT_E("AT24CXX i2c invalid.");
        return;
    }

    for (i = 0; i < 8; i++)
    {
        write_buff[i] = (os_uint8_t)rand();
    }

    write_regs(at24cxx_i2c, reg, write_buff, sizeof(write_buff));    
    os_task_mdelay(50);

    memset(read_buff, 0, sizeof(read_buff));
    read_regs(at24cxx_i2c, reg, read_buff, sizeof(read_buff));

    for (i = 0; i < 8; i++)
    {
        os_kprintf("write: 0x%02x, read: 0x%02x\r\n", write_buff[i], read_buff[i]);
    
        if (read_buff[i] != write_buff[i])
        {
            LOG_EXT_E("at24xx test failed index:%d, write:0x%02x, read:0x%02x", i, write_buff[i], read_buff[i]);
            return;
        }
    }

    os_kprintf("at24xx test success.\r\n");
    os_free(at24cxx_i2c);
}

static void at24cxx_test(int argc, char **argv)
{
    if (argc != 2)
    {
        LOG_EXT_E("at24cxx_test <i2c bus name>");
        LOG_EXT_E("eg: at24cxx_test i2c2");
        return;
    }

    __at24cxx_test(argv[1]);
}

SH_CMD_EXPORT(at24cxx_test, at24cxx_test,"test_at24cxx!");


