/**
 ***********************************************************************************************************************
 * Copyright (c) 2020, China Mobile Communications Group Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the \"License\ you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an \"AS IS\" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * \@file        drv_gpio.c
 *
 * \@brief       This file implements gpio driver for FM33A0xx.
 *
 * \@revision
 * Date         Author          Notes
 * 2020-02-20   OneOS Team      First Version
 ***********************************************************************************************************************
 */

#include "board.h"
#include <drv_cfg.h>
#include <os_hw.h>
#include <os_device.h>
#include <os_irq.h>
#include <os_memory.h>
#include "drv_gpio.h"
#include "drv_common.h"

#ifdef OS_USING_PIN

#define GPIOA
#define GPIOB

static const struct pin_index pins[] =
{
#if defined(GPIOA)
    _ATSAMD21_PIN(0,  PA00),
    _ATSAMD21_PIN(1,  PA01),
    _ATSAMD21_PIN(2,  PA02),
    _ATSAMD21_PIN(3,  PA03),
    _ATSAMD21_PIN(4,  PA04),
    _ATSAMD21_PIN(5,  PA05),
    _ATSAMD21_PIN(6,  PA06),
    _ATSAMD21_PIN(7,  PA07),
    _ATSAMD21_PIN(8,  PA08),
    _ATSAMD21_PIN(9,  PA09),
    _ATSAMD21_PIN(10, PA10),
    _ATSAMD21_PIN(11, PA11),
    _ATSAMD21_PIN(12, PA12),
    _ATSAMD21_PIN(13, PA13),
    _ATSAMD21_PIN(14, PA14),
    _ATSAMD21_PIN(15, PA15),
    _ATSAMD21_PIN(16, PA16),
    _ATSAMD21_PIN(17, PA17),
    _ATSAMD21_PIN(18, PA18),
    _ATSAMD21_PIN(19, PA19),
    _ATSAMD21_PIN(20, PA20),
    _ATSAMD21_PIN(21, PA21),
    _ATSAMD21_PIN(22, PA22),
    _ATSAMD21_PIN(23, PA23),
    _ATSAMD21_PIN(24, PA24),
    _ATSAMD21_PIN(25, PA25),
    _ATSAMD21_PIN(26, PA26),
    _ATSAMD21_PIN(27, PA27),
    _ATSAMD21_PIN(28, PA28),
    _ATSAMD21_PIN(29, PA29),
    _ATSAMD21_PIN(30, PA30),
    _ATSAMD21_PIN(31, PA31),
#endif

#if defined(GPIOB)
    _ATSAMD21_PIN(32,  PB00),
    _ATSAMD21_PIN(33,  PB01),
    _ATSAMD21_PIN(34,  PB02),
    _ATSAMD21_PIN(35,  PB03),
    _ATSAMD21_PIN(36,  PB04),
    _ATSAMD21_PIN(37,  PB05),
    _ATSAMD21_PIN(38,  PB06),
    _ATSAMD21_PIN(39,  PB07),
    _ATSAMD21_PIN(40,  PB08),
    _ATSAMD21_PIN(41,  PB09),
    _ATSAMD21_PIN(42, PB10),
    _ATSAMD21_PIN(43, PB11),
    _ATSAMD21_PIN(44, PB12),
    _ATSAMD21_PIN(45, PB13),
    _ATSAMD21_PIN(46, PB14),
    _ATSAMD21_PIN(47, PB15),
    _ATSAMD21_PIN(48, PB16),
    _ATSAMD21_PIN(49, PB17),
    _ATSAMD21_PIN(50, PB18),
    _ATSAMD21_PIN(51, PB19),
    _ATSAMD21_PIN(52, PB20),
    _ATSAMD21_PIN(53, PB21),
    _ATSAMD21_PIN(54, PB22),
    _ATSAMD21_PIN(55, PB23),
    _ATSAMD21_PIN(56, PB24),
    _ATSAMD21_PIN(57, PB25),
    _ATSAMD21_PIN(58, PB26),
    _ATSAMD21_PIN(59, PB27),
    _ATSAMD21_PIN(60, PB28),
    _ATSAMD21_PIN(61, PB29),
    _ATSAMD21_PIN(62, PB30),
    _ATSAMD21_PIN(63, PB31),
#endif

};


#define ITEM_NUM(items) sizeof(items) / sizeof(items[0])

os_uint32_t os_pin_get(os_uint32_t port_pin)
{
    os_uint32_t i;

    for (i = 0; i < ITEM_NUM(pins); i++)
    {
        if (pins[i].port_pin == port_pin)
        {
            return pins[i].index;
        }
    }

    return 0xffff;
}


static const struct pin_index *get_pin(os_uint8_t pin)
{
    const struct pin_index *index;

    if (pin < ITEM_NUM(pins))
    {
        index = &pins[pin];
    }
    else
    {
        index = OS_NULL;
    }

    return index;
};

void samd21_pin_mode(struct os_device *dev, os_base_t pin, os_base_t mode)
{
    const struct pin_index *index;
    enum gpio_pull_mode pull_mode;
    enum gpio_direction direction;
    os_uint8_t port_pin;

    index = get_pin(pin);
    if (index == OS_NULL)
    {
        return;
    }

    port_pin = index->port_pin;

    if (mode == PIN_MODE_OUTPUT)
    {
        pull_mode = GPIO_PULL_OFF;
        direction = GPIO_DIRECTION_OUT;
    }
    else  if (mode == PIN_MODE_OUTPUT_OD)
    {
        /* output setting */
        pull_mode = GPIO_PULL_OFF;
        direction = GPIO_DIRECTION_OUT;
    }
    else if (mode == PIN_MODE_INPUT)
    {
        /* input setting: not pull. */
        pull_mode = GPIO_PULL_OFF;
        direction = GPIO_DIRECTION_IN;
    }
    else if (mode == PIN_MODE_INPUT_PULLUP)
    {
        /* input setting: pull up. */
        pull_mode = GPIO_PULL_UP;
        direction = GPIO_DIRECTION_IN;
    }
    else if (mode == PIN_MODE_INPUT_PULLDOWN)
    {
        pull_mode = GPIO_PULL_DOWN;
        direction = GPIO_DIRECTION_IN;
    }

    gpio_set_pin_pull_mode(port_pin, pull_mode);
    gpio_set_pin_direction(port_pin, direction);

}

void samd21_pin_write(struct os_device *dev, os_base_t pin, os_base_t value)
{
    const struct pin_index *index;

    index = get_pin(pin);
    if (index == OS_NULL)
    {
        return;
    }

    if (value == PIN_LOW)
    {
        gpio_set_pin_level(index->port_pin, false);
    }
    else if (value == PIN_HIGH)
    {
        gpio_set_pin_level(index->port_pin, true);
    }
}

os_int32_t samd21_pin_read(struct os_device *dev, os_base_t pin)
{
    os_int32_t value = PIN_LOW;
    const struct pin_index *index;

    index = get_pin(pin);
    if (index == OS_NULL)
    {
        return value;
    }

    value = gpio_get_pin_level(index->port_pin);

    return value;
}




const static struct os_pin_ops am_pin_ops =
{
    samd21_pin_mode,
    samd21_pin_write,
    samd21_pin_read,
//    samd21_pin_attach_irq,
//    samd21_pin_dettach_irq,
//    samd21_pin_irq_enable,
};




os_int32_t os_hw_pin_init(void)
{
//    os_int32_t i = 0;

//    for (i = 0; i < FM33A0XX_IRQ_NUMBERS; i++)
//        samd21_pin_irq_hdr_tab[i].pin = -1;

    os_device_pin_register(0, &am_pin_ops, OS_NULL);

    os_kprintf("pin_init!\n");

    return 0;
}


/* INIT_BOARD_EXPORT(os_hw_pin_init); */
#endif

