/**
 ***********************************************************************************************************************
 * Copyright (c) 2020, China Mobile Communications Group Co.,Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with 
 * the License. You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * @file        drv_usart.c
 *
 * @brief       This file implements usart driver for stm32
 *
 * @revision
 * Date         Author          Notes
 * 2020-02-20   OneOS Team      First Version
 ***********************************************************************************************************************
 */

#include <board.h>
#include <os_irq.h>
#include <os_memory.h>
#include <bus/bus.h>
#include <hal_gpio.h>
#include <drv_gpio.h>
#include <hpl_usart_sync.h>
#include <drv_uart.h>

#define DRV_EXT_LVL DBG_EXT_INFO
#define DRV_EXT_TAG "drv.usart"
#include <drv_log.h>


#ifdef OS_USING_SERIAL

static os_uint8_t USART_x_buffer[OS_SERIAL_RX_BUFSZ];

struct samd21_uart
{
    struct os_serial_device serial;

    UART_HandleTypeDef *huart;

    os_list_node_t list;
};

static os_list_node_t samd21_uart_list = OS_LIST_INIT(samd21_uart_list);

void HAL_UART_TxCpltCallback(UART_HandleTypeDef *huart)
{
    struct samd21_uart *uart;

    os_list_for_each_entry(uart, &samd21_uart_list, struct samd21_uart, list)
    {
        if (uart->huart == huart)
        {
            os_interrupt_enter();
            os_hw_serial_isr_txdone((struct os_serial_device *)uart);
            os_interrupt_leave();
            break;
        }
    }
}

void HAL_UART_RxCpltCallback(UART_HandleTypeDef *huart)
{
    struct samd21_uart *uart;
    os_size_t num;
    

    os_list_for_each_entry(uart, &samd21_uart_list, struct samd21_uart, list)
    {
        if (uart->huart == huart)
        {
            os_interrupt_enter();
            num = ringbuffer_num(&huart->usart_xx.rx);
            num = io_read(uart->huart->io, huart->pRxBuffPtr, num);
            os_hw_serial_isr_rxdone((struct os_serial_device *)uart, num);
            os_interrupt_leave();
            break;
        }
    }
}


static void tx_cb_USART_x(const struct usart_async_descriptor *const io_descr)
{
    UART_HandleTypeDef *huart = NULL;
    
    huart = (UART_HandleTypeDef *)io_descr;
    HAL_UART_TxCpltCallback(huart);
}

static void rx_cb_USART_x(const struct usart_async_descriptor *const io_descr)
{
    UART_HandleTypeDef *huart = NULL;
    
    huart = (UART_HandleTypeDef *)io_descr;
    HAL_UART_RxCpltCallback(huart);
}

#ifndef COM_BAUD
#define COM_BAUD(baud)                                                                                  \
	65536 - ((65536 * 16.0f * baud) / CONF_GCLK_SERCOM5_CORE_FREQUENCY)
#endif

static os_err_t samd21_cfg(UART_HandleTypeDef * huart)
{
    UART_InitTypeDef *init_cfg = NULL;

    init_cfg = &(huart->Init);
    
    _pm_enable_bus_clock(init_cfg->bus, init_cfg->sercom);
    _gclk_enable_channel(init_cfg->sercom_core_id, init_cfg->sercom_core_src);

    usart_async_init(&huart->usart_xx, init_cfg->sercom, USART_x_buffer, sizeof(USART_x_buffer), (void *)NULL);

    gpio_set_pin_function(init_cfg->com_tx.pin_num, init_cfg->com_tx.pin_func);
    gpio_set_pin_function(init_cfg->com_rx.pin_num, init_cfg->com_rx.pin_func);

    usart_async_set_stopbits(&huart->usart_xx, huart->Init.StopBits);
    usart_async_set_parity(&huart->usart_xx, huart->Init.Parity);
    usart_async_set_character_size(&huart->usart_xx, huart->Init.WordLength);

    usart_async_set_baud_rate(&huart->usart_xx, COM_BAUD(init_cfg->baud));

    usart_async_get_io_descriptor(&huart->usart_xx, &huart->io);
    usart_async_enable(&huart->usart_xx);

    return OS_EOK;
}

static os_err_t samd21_configure(struct os_serial_device *serial, struct serial_configure *cfg)
{
    struct samd21_uart *uart = NULL;
    
    OS_ASSERT(serial != OS_NULL);
    OS_ASSERT(cfg != OS_NULL);
    
    uart = os_container_of(serial, struct samd21_uart, serial);
    OS_ASSERT(uart!= OS_NULL);
    OS_ASSERT(uart->huart!= OS_NULL);
    
    uart->huart->Init.baud = cfg->baud_rate;
    
    switch (cfg->stop_bits)
    {
    case STOP_BITS_1:
        uart->huart->Init.StopBits = USART_STOP_BITS_ONE;
        break;
    case STOP_BITS_2:
        uart->huart->Init.StopBits = USART_STOP_BITS_TWO;
        break;
    default:
        return OS_EINVAL;
    }
    
    switch (cfg->parity)
    {
    case PARITY_NONE:
        uart->huart->Init.Parity = USART_PARITY_NONE;
        break;
    case PARITY_ODD:
        uart->huart->Init.Parity = USART_PARITY_ODD;
        break;
    case PARITY_EVEN:
        uart->huart->Init.Parity = USART_PARITY_EVEN;
        break;
    default:
        return OS_EINVAL;
    }

    switch (cfg->data_bits)
    {
    case DATA_BITS_5:
        uart->huart->Init.WordLength = USART_CHARACTER_SIZE_5BITS;
        break;
    case DATA_BITS_6:
        uart->huart->Init.WordLength = USART_CHARACTER_SIZE_6BITS;
        break;
    case DATA_BITS_7:
        uart->huart->Init.WordLength = USART_CHARACTER_SIZE_7BITS;
        break;
    case DATA_BITS_8:
        uart->huart->Init.WordLength = USART_CHARACTER_SIZE_8BITS;
        break;
    case DATA_BITS_9:
        uart->huart->Init.WordLength = USART_CHARACTER_SIZE_9BITS;
        break;
    default:
        return OS_EINVAL;
    }

    samd21_cfg(uart->huart);

    return OS_EOK;
}

void HAL_UART_Transmit_IT(UART_HandleTypeDef *huart, const os_uint8_t *pData, os_uint16_t Size)
{
    usart_async_register_callback(&huart->usart_xx, USART_ASYNC_TXC_CB, tx_cb_USART_x);
}


static int samd21_uart_start_send(struct os_serial_device *serial, const os_uint8_t *buff, os_size_t size)
{
    struct samd21_uart *uart;
    os_size_t  ret;
    
    OS_ASSERT(serial != OS_NULL);
    OS_ASSERT(buff != OS_NULL);

    uart = os_container_of(serial, struct samd21_uart, serial);
    OS_ASSERT(uart!= OS_NULL);
    OS_ASSERT(uart->huart!= OS_NULL);

    HAL_UART_Transmit_IT(uart->huart, buff, size);

    ret = io_write(uart->huart->io, (uint8_t *)buff, size);        

    return ret;
}

static int samd21_uart_stop_send(struct os_serial_device *serial)
{
    struct samd21_uart *uart;
    
    OS_ASSERT(serial != OS_NULL);

    uart = os_container_of(serial, struct samd21_uart, serial);
    OS_ASSERT(uart!= OS_NULL);
    OS_ASSERT(uart->huart!= OS_NULL);
    
    return 0;
}


void HAL_UART_Receive_IT(UART_HandleTypeDef *huart, os_uint8_t *pData, os_uint16_t Size)
{    
    huart->pRxBuffPtr  = pData;
    huart->RxXferSize  = Size;
    huart->RxXferCount = Size;
    
    usart_async_register_callback(&huart->usart_xx, USART_ASYNC_RXC_CB, rx_cb_USART_x);
}

static int samd21_uart_start_recv(struct os_serial_device *serial, os_uint8_t *buff, os_size_t size)
{
    os_size_t  ret;
    struct samd21_uart *uart;

    OS_ASSERT(serial != OS_NULL);

    uart = os_container_of(serial, struct samd21_uart, serial);    
    OS_ASSERT(uart!= OS_NULL);
    OS_ASSERT(uart->huart!= OS_NULL);

    HAL_UART_Receive_IT(uart->huart, buff, size);
    
    ret = io_read(uart->huart->io, (uint8_t *)buff, size);

    return ret;
}

static int samd21_uart_stop_recv(struct os_serial_device *serial)
{
    struct samd21_uart *uart;
    
    OS_ASSERT(serial != OS_NULL);

    uart = os_container_of(serial, struct samd21_uart, serial);
    OS_ASSERT(uart!= OS_NULL);
    OS_ASSERT(uart->huart!= OS_NULL);

    return 0;
}

static int samd21_uart_recv_state(struct os_serial_device *serial)
{
    int state;
    struct samd21_uart *uart;
    
    OS_ASSERT(serial != OS_NULL);

    uart = os_container_of(serial, struct samd21_uart, serial);
    OS_ASSERT(uart!= OS_NULL);
    OS_ASSERT(uart->huart!= OS_NULL);

    state = ringbuffer_num(&uart->huart->usart_xx.rx);
    state |= OS_SERIAL_FLAG_RX_IDLE;

    return state;
}

bool samd21_usart_async_is_ready_to_send(const struct _usart_async_device *const device)
{
	return hri_sercomusart_get_interrupt_DRE_bit(device->hw);
}

/**
 * \brief Check if USART transmission complete
 */
bool samd21_usart_async_is_transmit_done(const struct _usart_async_device *const device)
{
	return hri_sercomusart_get_interrupt_TXC_bit(device->hw);
}

static os_int32_t samd21_usart_sync_write(struct io_descriptor *const io_descr, const uint8_t *const buf, const uint16_t length)
{
	os_uint32_t                      offset = 0;
	struct usart_async_descriptor *descr  = (struct usart_async_descriptor *)io_descr;

	ASSERT(io_descr && buf && length);
	while (!samd21_usart_async_is_ready_to_send(&descr->device))
		;
	do {
		_usart_async_write_byte(&descr->device, buf[offset]);
		while (!samd21_usart_async_is_ready_to_send(&descr->device))
			;
	} while (++offset < length);
	while (!samd21_usart_async_is_transmit_done(&descr->device))
		;
	return (os_int32_t)offset;
}

os_uint8_t samd21_usart_async_read_byte(const struct _usart_async_device *const device)
{
	return hri_sercomusart_read_DATA_reg(device->hw);
}

bool samd21_usart_async_is_byte_received(const struct _usart_async_device *const device)
{
	return hri_sercomusart_get_interrupt_RXC_bit(device->hw);
}


static os_int32_t samd21_usart_sync_read(struct io_descriptor *const io_descr, uint8_t *const buf, const uint16_t length)
{
	os_uint32_t                      offset = 0;
	struct usart_async_descriptor *descr  = (struct usart_async_descriptor *)io_descr;

	ASSERT(io_descr && buf && length);
	do {
		while (!samd21_usart_async_is_byte_received(&descr->device))
			;
		buf[offset] = samd21_usart_async_read_byte(&descr->device);
	} while (++offset < length);

	return (os_int32_t)offset;
}


static int samd21_uart_poll_send(struct os_serial_device *serial, const os_uint8_t *buff, os_size_t size)
{
    struct samd21_uart *uart = NULL;
    struct io_descriptor *io = NULL;

    int i;
    os_base_t level;
    
    OS_ASSERT(serial != OS_NULL);

    uart = os_container_of(serial, struct samd21_uart, serial);
    OS_ASSERT(uart!= OS_NULL);
    OS_ASSERT(uart->huart!= OS_NULL);

    io = uart->huart->io;

    for (i = 0; i < size; i++)
    {
        level = os_hw_interrupt_disable();
        samd21_usart_sync_write(io, buff, 1);

        os_hw_interrupt_enable(level);
    }

    return size;
}

static int samd21_uart_poll_recv(struct os_serial_device *serial, os_uint8_t *buff, os_size_t size)
{
    struct samd21_uart *uart;
    os_size_t  ret;
    struct io_descriptor *io = NULL;

    os_base_t level;
    
    OS_ASSERT(serial != OS_NULL);
    OS_ASSERT(size == 1);

    uart = os_container_of(serial, struct samd21_uart, serial);
    io = uart->huart->io;

    level = os_hw_interrupt_disable();
    ret = samd21_usart_sync_read(io, buff, size);
    os_hw_interrupt_enable(level);

    return ret;
}

static const struct os_uart_ops samd21_uart_ops = {
    .configure    = samd21_configure,

    .start_send   = samd21_uart_start_send,
    .stop_send    = samd21_uart_stop_send,

    .start_recv   = samd21_uart_start_recv,
    .stop_recv    = samd21_uart_stop_recv,
    .recv_state   = samd21_uart_recv_state,
    
    .poll_send    = samd21_uart_poll_send,
    .poll_recv    = samd21_uart_poll_recv,
};


static int samd21_usart_probe(const os_driver_info_t *drv, const os_device_info_t *dev)
{
    struct serial_configure config  = OS_SERIAL_CONFIG_DEFAULT;
    
    os_err_t    result  = 0;
    os_base_t   level;

    struct samd21_uart *uart = os_calloc(1, sizeof(struct samd21_uart));

    OS_ASSERT(uart);

    uart->huart = (UART_HandleTypeDef *)dev->info;

    struct os_serial_device *serial = &uart->serial;

    serial->ops    = &samd21_uart_ops;
    serial->config = config;

    level = os_hw_interrupt_disable();
    os_list_add_tail(&samd21_uart_list, &uart->list);
    os_hw_interrupt_enable(level);
    
    result = os_hw_serial_register(serial, dev->name, OS_DEVICE_FLAG_RDWR, NULL);
    
    OS_ASSERT(result == OS_EOK);

    return result;
}

#else   /* OS_USING_SERIAL */

#endif

OS_DRIVER_INFO samd21_usart_driver = {
    .name   = "UART_HandleTypeDef",
    .probe  = samd21_usart_probe,
};

OS_DRIVER_DEFINE(samd21_usart_driver, "0.end.0");

