#include <board.h>
#include <drv_cfg.h>
#include <os_hw.h>
#include <os_memory.h>
#include <sensors/sensor.h>
#include <drv_log.h>
#include <string.h>
#include <os_mailbox.h>

#ifdef OS_USING_SHELL
#include <shell.h>
#endif

/**************************************************************************
    I2C ADDRESS/BITS
**************************************************************************/
#define LIS3DHTR_DEFAULT_ADDRESS (0x18) // 3C >> 1 = 7-bit default
#define LIS3DHTR_ADDRESS_UPDATED (0x19) //

/**************************************************************************
    ACCELEROMETER REGISTERS
**************************************************************************/
#define LIS3DHTR_REG_TEMP_CFG (0x1F)            // Temperature Sensor Register
#define LIS3DHTR_REG_ACCEL_CTRL_REG1 (0x20)     // Accelerometer Control Register 1
#define LIS3DHTR_REG_ACCEL_CTRL_REG4 (0x23)     // Accelerometer Control Register 4
#define LIS3DHTR_REG_ACCEL_OUT_X_L (0x28)       // X-Axis Acceleration Data Low Register
#define LIS3DHTR_REG_ACCEL_OUT_X_H (0x29)       // X-Axis Acceleration Data High Register
#define LIS3DHTR_REG_ACCEL_OUT_Y_L (0x2A)       // Y-Axis Acceleration Data Low Register
#define LIS3DHTR_REG_ACCEL_OUT_Y_H (0x2B)       // Y-Axis Acceleration Data High Register
#define LIS3DHTR_REG_ACCEL_OUT_Z_L (0x2C)       // Z-Axis Acceleration Data Low Register
#define LIS3DHTR_REG_ACCEL_OUT_Z_H (0x2D)       // Z-Axis Acceleration Data High Register

/**************************************************************************
    TEMPERATURE REGISTER DESCRIPTION
**************************************************************************/
#define LIS3DHTR_REG_TEMP_ADC_PD_MASK (0x80)     // ADC Power Enable Status
#define LIS3DHTR_REG_TEMP_ADC_PD_DISABLED (0x00) // ADC Disabled
#define LIS3DHTR_REG_TEMP_ADC_PD_ENABLED (0x80)  // ADC Enabled

#define LIS3DHTR_REG_TEMP_TEMP_EN_MASK (0x40)     // Temperature Sensor (T) Enable Status
#define LIS3DHTR_REG_TEMP_TEMP_EN_DISABLED (0x00) // Temperature Sensor (T) Disabled
#define LIS3DHTR_REG_TEMP_TEMP_EN_ENABLED (0x40)  // Temperature Sensor (T) Enabled

/**************************************************************************
    ACCELEROMETER CONTROL REGISTER 1 DESCRIPTION
**************************************************************************/
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_MASK (0xF0) // Acceleration Data Rate Selection
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_PD (0x00)   // Power-Down Mode
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_1 (0x10)    // Normal / Low Power Mode (1 Hz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_10 (0x20)   // Normal / Low Power Mode (10 Hz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_25 (0x30)   // Normal / Low Power Mode (25 Hz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_50 (0x40)   // Normal / Low Power Mode (50 Hz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_100 (0x50)  // Normal / Low Power Mode (100 Hz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_200 (0x60)  // Normal / Low Power Mode (200 Hz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_400 (0x70)  // Normal / Low Power Mode (400 Hz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_1_6K (0x80) // Low Power Mode (1.6 KHz)
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_5K (0x90)   // Normal (1.25 KHz) / Low Power Mode (5 KHz)

#define LIS3DHTR_REG_ACCEL_CTRL_REG1_LPEN_MASK (0x08)   // Low Power Mode Enable
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_LPEN_NORMAL (0x00) // Normal Mode
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_LPEN_LOW (0x08)    // Low Power Mode

#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AZEN_MASK (0x04)    // Acceleration Z-Axis Enable
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AZEN_DISABLE (0x00) // Acceleration Z-Axis Disabled
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AZEN_ENABLE (0x04)  // Acceleration Z-Axis Enabled

#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AYEN_MASK (0x02)    // Acceleration Y-Axis Enable
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AYEN_DISABLE (0x00) // Acceleration Y-Axis Disabled
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AYEN_ENABLE (0x02)  // Acceleration Y-Axis Enabled

#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AXEN_MASK (0x01)    // Acceleration X-Axis Enable
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AXEN_DISABLE (0x00) // Acceleration X-Axis Disabled
#define LIS3DHTR_REG_ACCEL_CTRL_REG1_AXEN_ENABLE (0x01)  // Acceleration X-Axis Enabled

/**************************************************************************
    ACCELEROMETER CONTROL REGISTER 4 DESCRIPTION
**************************************************************************/
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_BDU_MASK (0x80)       // Block Data Update
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_BDU_CONTINUOUS (0x00) // Continuous Update
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_BDU_NOTUPDATED (0x80) // Output Registers Not Updated until MSB and LSB Read

#define LIS3DHTR_REG_ACCEL_CTRL_REG4_BLE_MASK (0x40) // Big/Little Endian Data Selection
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_BLE_LSB (0x00)  // Data LSB @ lower address
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_BLE_MSB (0x40)  // Data MSB @ lower address

#define LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_MASK (0x30) // Full-Scale Selection
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_2G (0x00)   // +/- 2G
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_4G (0x10)   // +/- 4G
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_8G (0x20)   // +/- 8G
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_16G (0x30)  // +/- 16G

#define LIS3DHTR_REG_ACCEL_CTRL_REG4_HS_MASK (0x08)    // High Resolution Output Mode
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_HS_DISABLE (0x00) // High Resolution Disable
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_HS_ENABLE (0x08)  // High Resolution Enable

#define LIS3DHTR_REG_ACCEL_CTRL_REG4_ST_MASK (0x06)   // Self-Test Enable
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_ST_NORMAL (0x00) // Normal Mode
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_ST_0 (0x02)      // Self-Test 0
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_ST_1 (0x04)      // Self-Test 1

#define LIS3DHTR_REG_ACCEL_CTRL_REG4_SIM_MASK (0x01)  // SPI Serial Interface Mode Selection
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_SIM_4WIRE (0x00) // 4-Wire Interface
#define LIS3DHTR_REG_ACCEL_CTRL_REG4_SIM_3WIRE (0x01) // 3-Wire Interface

#define LIS3DHTR_REG_ACCEL_STATUS2_UPDATE_MASK (0x08)   // Has New Data Flag Mask

enum power_type_t // power mode
{
    POWER_MODE_NORMAL = LIS3DHTR_REG_ACCEL_CTRL_REG1_LPEN_NORMAL,
    POWER_MODE_LOW = LIS3DHTR_REG_ACCEL_CTRL_REG1_LPEN_LOW
};

enum scale_type_t // measurement rage
{
    LIS3DHTR_RANGE_2G = LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_2G,   //
    LIS3DHTR_RANGE_4G = LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_4G,   //
    LIS3DHTR_RANGE_8G = LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_8G,   //
    LIS3DHTR_RANGE_16G = LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_16G, //
};

enum odr_type_t // output data rate
{
    LIS3DHTR_DATARATE_POWERDOWN = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_PD,
    LIS3DHTR_DATARATE_1HZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_1,
    LIS3DHTR_DATARATE_10HZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_10,
    LIS3DHTR_DATARATE_25HZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_25,
    LIS3DHTR_DATARATE_50HZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_50,
    LIS3DHTR_DATARATE_100HZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_100,
    LIS3DHTR_DATARATE_200HZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_200,
    LIS3DHTR_DATARATE_400HZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_400,
    LIS3DHTR_DATARATE_1_6KH = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_1_6K,
    LIS3DHTR_DATARATE_5KHZ = LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_5K
};

static struct os_i2c_bus_device* acc_i2c_bus;

os_mailbox_t *mb_iic_LIS = NULL;
#define MB_MAX_MAILS    10

typedef struct
{
    os_uint32_t x;
    os_uint32_t y;
    os_uint32_t z;
}x_y_z_s;

static os_err_t write_regs(struct os_i2c_bus_device *bus, os_uint16_t reg, os_uint8_t *buf, os_uint8_t len)
{
    struct os_i2c_msg msgs;
    os_uint8_t        databuf[9];
    os_uint8_t        device_addr = 0;
    os_uint8_t        reg_addr    = 0;

    OS_ASSERT(len <= 8);

    if (reg > 255)
    {
        device_addr = LIS3DHTR_ADDRESS_UPDATED | 0x01;
    }
    else
    {
        device_addr = LIS3DHTR_ADDRESS_UPDATED;
    }
    reg_addr   = reg & 0xFF;
    databuf[0] = reg_addr;

    memcpy(&databuf[1], buf, len);

    msgs.addr  = device_addr;
    msgs.flags = OS_I2C_WR;
    msgs.buf   = databuf;
    msgs.len   = len + 1;

    if (os_i2c_transfer(bus, &msgs, 1) == 1)
    {
        return OS_EOK;
    }
    else
    {
        LOG_EXT_E("Writing command error");
        return -OS_ERROR;
    }
}

static os_err_t read_regs(struct os_i2c_bus_device *bus, os_uint16_t reg, os_uint8_t *buf, os_uint8_t len)
{
    struct os_i2c_msg msgs[2];
    os_uint8_t        device_addr = 0;
    os_uint8_t        reg_addr    = 0;

    if (reg > 255)
    {
        device_addr = LIS3DHTR_ADDRESS_UPDATED | 0x01;
    }
    else
    {
        device_addr = LIS3DHTR_ADDRESS_UPDATED;
    }
    reg_addr = reg & 0xFF;

    msgs[0].addr  = device_addr;
    msgs[0].flags = OS_I2C_WR;
    msgs[0].buf   = &reg_addr;
    msgs[0].len   = 1;

    msgs[1].addr  = device_addr;
    msgs[1].flags = OS_I2C_RD;
    msgs[1].buf   = buf;
    msgs[1].len   = len;

    if (os_i2c_transfer(bus, msgs, 2) == 2)
    {
        return OS_EOK;
    }
    else
    {
        LOG_EXT_E("Reading command error");
        return -OS_ERROR;
    }
}

void __lis3dhtr_test1(const char* i2c_bus)
{
    int        i;
    os_uint8_t read_buff[8];
    os_uint8_t write_buff[8];

    /* find i2c device  */
    acc_i2c_bus = (struct os_i2c_bus_device *)os_device_find(i2c_bus);
    if (acc_i2c_bus == NULL)
    {
        LOG_EXT_E("LIS3DHTR i2c invalid.");
        return;
    }
	
	uint8_t config5 = LIS3DHTR_REG_TEMP_ADC_PD_ENABLED |
                      LIS3DHTR_REG_TEMP_TEMP_EN_DISABLED;

	write_regs(acc_i2c_bus, LIS3DHTR_REG_TEMP_CFG, &config5, sizeof(config5));
	
    os_task_mdelay(100);

    uint8_t config1 = LIS3DHTR_REG_ACCEL_CTRL_REG1_LPEN_NORMAL | // Normal Mode
                      LIS3DHTR_REG_ACCEL_CTRL_REG1_AZEN_ENABLE | // Acceleration Z-Axis Enabled
                      LIS3DHTR_REG_ACCEL_CTRL_REG1_AYEN_ENABLE | // Acceleration Y-Axis Enabled
                      LIS3DHTR_REG_ACCEL_CTRL_REG1_AXEN_ENABLE;
	
	write_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG1, &config1, sizeof(config1));
    os_task_mdelay(100);

    uint8_t config4 = LIS3DHTR_REG_ACCEL_CTRL_REG4_BDU_NOTUPDATED | // Continuous Update
                      LIS3DHTR_REG_ACCEL_CTRL_REG4_BLE_LSB |        // Data LSB @ lower address
                      LIS3DHTR_REG_ACCEL_CTRL_REG4_HS_DISABLE |      // High Resolution Disable
                      LIS3DHTR_REG_ACCEL_CTRL_REG4_ST_NORMAL |      // Normal Mode
                      LIS3DHTR_REG_ACCEL_CTRL_REG4_SIM_4WIRE;       // 4-Wire Interface

	write_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG4, &config4, sizeof(config4));
	
	uint8_t data = 0;

	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG4, &data, sizeof(data));

    data &= ~LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_MASK;
    data |= LIS3DHTR_REG_ACCEL_CTRL_REG4_FS_16G;

	write_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG4, &data, sizeof(data));
    os_task_mdelay(100);
	
    uint8_t data_1 = 0;

	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG1, &data_1, sizeof(data_1));

    data_1 &= ~LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_MASK;
    data_1 |= LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_400;

	write_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG1, &data_1, sizeof(data_1));
    os_task_mdelay(100);
	
    uint8_t data_2 = 0;

    read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG1, &data_2, sizeof(data_2));

    data_2 &= ~LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_MASK;
    data_2 |= LIS3DHTR_REG_ACCEL_CTRL_REG1_AODR_50;

    write_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG1, &data_2, sizeof(data_2));
    os_task_mdelay(100);

    uint8_t data_3 = 0;
    read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG4, &data_3, sizeof(data_3));

    data_3 = LIS3DHTR_REG_ACCEL_CTRL_REG4_HS_ENABLE;         //data & ~LIS3DHTR_REG_ACCEL_CTRL_REG4_HS_ENABLE;
    write_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_CTRL_REG4, &data_3, sizeof(data_3));
	

}

void entry_mb_tid3(void * parameter)
{
	// Read the Accelerometer
    uint8_t xAccelLo, xAccelHi, yAccelLo, yAccelHi, zAccelLo, zAccelHi;
	
    x_y_z_s *x_y_z = NULL;
    x_y_z = (x_y_z_s *)os_malloc(sizeof(x_y_z_s));
    ASSERT(x_y_z != NULL);

    while (1)
    {
    os_task_msleep(1000);
    // Read the Data
    // Reading the Low X-Axis Acceleration Data Register
	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_OUT_X_L, &xAccelLo, sizeof(xAccelLo));
    // Reading the High X-Axis Acceleration Data Register
	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_OUT_X_H, &xAccelHi, sizeof(xAccelHi));
    // Conversion of the result
    // 16-bit signed result for X-Axis Acceleration Data of LIS3DHTR
    x_y_z->x = ((int16_t)((xAccelHi << 8) | xAccelLo))/1280;

    // Reading the Low Y-Axis Acceleration Data Register
	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_OUT_Y_L, &yAccelLo, sizeof(yAccelLo));
    // Reading the High Y-Axis Acceleration Data Register
	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_OUT_Y_H, &yAccelHi, sizeof(yAccelHi));
    // Conversion of the result
    // 16-bit signed result for Y-Axis Acceleration Data of LIS3DHTR
    x_y_z->y = ((int16_t)((yAccelHi << 8) | yAccelLo))/1280;

    // Reading the Low Z-Axis Acceleration Data Register
	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_OUT_Z_L, &zAccelLo, sizeof(zAccelLo));
    // Reading the High Z-Axis Acceleration Data Register
	read_regs(acc_i2c_bus, LIS3DHTR_REG_ACCEL_OUT_Z_H, &zAccelHi, sizeof(zAccelHi));
    // Conversion of the result
    // 16-bit signed result for Z-Axis Acceleration Data of LIS3DHTR
    x_y_z->z = ((int16_t)((zAccelHi << 8)) | zAccelLo)/1280;

    if(x_y_z->x == 65535){
        x_y_z->x = 2;
    }
    if(x_y_z->y == 65535){
        x_y_z->y = 2;
    }
    if(x_y_z->z == 65535){
        x_y_z->z = 2;
    }


    os_kprintf("start send: LIS3DHTD data:x[%d], y[%d], z[%d]\r\n", x_y_z->x, x_y_z->y, x_y_z->z);
    os_mb_send(mb_iic_LIS, (os_uint32_t)x_y_z, OS_IPC_WAITING_FOREVER);


    }
    

}


void entry_mb_tid4(void * parameter)
{
    os_uint32_t data = 0;
    while (1)
    {
        os_mb_recv(mb_iic_LIS, &data, OS_IPC_WAITING_FOREVER);
        os_kprintf("recv LIS3DHTD data:x[%d], y[%d], z[%d]\r\n", ((x_y_z_s *)data)->x, ((x_y_z_s *)data)->y, ((x_y_z_s *)data)->z);
    }
}




static void LIS_mb_email(int argc, char** argv)
{
    os_task_t *tid1, *tid2;

    if (argc != 2)
    {
        LOG_EXT_E("lis3dhtr <i2c bus name>");
        LOG_EXT_E("eg: lis3dhtr soft_i2c1");
        return;
    }
 
    __lis3dhtr_test1(argv[1]);

    mb_iic_LIS = os_mb_create("mb", MB_MAX_MAILS, OS_IPC_FLAG_FIFO);
    if (mb_iic_LIS == NULL)
    {
        os_kprintf("os_mb_create err\r\n");
        return;
    }

    tid1 = os_task_create("mb_tid1", entry_mb_tid3, NULL, 512, 5, 10);
    OS_ASSERT(tid1 != NULL);
    os_task_startup(tid1);

    tid2 = os_task_create("mb_tid2", entry_mb_tid4, NULL, 512, 5, 10);
    OS_ASSERT(tid2 != NULL);
    os_task_startup(tid2);

    
}




#ifdef OS_USING_SHELL
SH_CMD_EXPORT(LIS_mb_email, LIS_mb_email,"LIS_mb_email");
#endif





// static void lis3dhtr_test(int argc, char** argv)
// {
//     if (argc != 2)
//     {
//         LOG_EXT_E("lis3dhtr <i2c bus name>");
//         LOG_EXT_E("eg: lis3dhtr soft_i2c1");
//         return;
//     }
 
//     __lis3dhtr_test(argv[1]);
// }


// #ifdef OS_USING_SHELL
// SH_CMD_EXPORT(lis3dhtr, LIS_mb_test,"lis3dhtr");
// #endif



