#include <board.h>
#include <os_clock.h>
#include <stdlib.h>
#include <oneos_config.h>
#include <os_dbg.h>
#include <os_errno.h>
#include <os_task.h>
#include <shell.h>
#include <os_sem.h>

// The timings are taken from Adafruit's NeoPixel library

#define PINOP(pin, OP) (PORT->Group[(pin) / 32].OP.reg = (1 << ((pin) % 32)))

static void neopixel_send_buffer_core(volatile uint32_t *clraddr, uint32_t pinMask,
                                      const uint8_t *ptr, int numBytes) __attribute__((naked));

static void neopixel_send_buffer_core(volatile uint32_t *clraddr, uint32_t pinMask,
                                      const uint8_t *ptr, int numBytes) {
    asm volatile("        push    {r4, r5, r6, lr};"
                 "        add     r3, r2, r3;"
                 "loopLoad:"
                 "        ldrb r5, [r2, #0];" // r5 := *ptr
                 "        add  r2, #1;"       // ptr++
                 "        movs    r4, #128;"  // r4-mask, 0x80
                 "loopBit:"
                 "        str r1, [r0, #4];"                    // set
                 "        movs r6, #3; d2: sub r6, #1; bne d2;" // delay 3
                 "        tst r4, r5;"                          // mask&r5
                 "        bne skipclr;"
                 "        str r1, [r0, #0];" // clr
                 "skipclr:"
                 "        movs r6, #6; d0: sub r6, #1; bne d0;" // delay 6
                 "        str r1, [r0, #0];"   // clr (possibly again, doesn't matter)
                 "        asr     r4, r4, #1;" // mask >>= 1
                 "        beq     nextbyte;"
                 "        uxtb    r4, r4;"
                 "        movs r6, #2; d1: sub r6, #1; bne d1;" // delay 2
                 "        b       loopBit;"
                 "nextbyte:"
                 "        cmp r2, r3;"
                 "        bcs stop;"
                 "        b loopLoad;"
                 "stop:"
                 "        pop {r4, r5, r6, pc};"
                 "");
}

// this assumes the pin has been configured correctly
static inline void neopixel_send_buffer(const uint8_t pinnum, const uint8_t *ptr, int numBytes) {
    uint8_t portNum = pinnum / 32;
    uint32_t pinMask = 1ul << (pinnum % 32);

    PINOP(pinnum, DIRSET);


    PINOP(pinnum, OUTCLR);
	
    os_task_msleep(1);

    volatile uint32_t *clraddr = &PORT->Group[portNum].OUTCLR.reg;

    // equivalent to cpu_irq_is_enabled()
    if (__get_PRIMASK() == 0) {
        __disable_irq();
        neopixel_send_buffer_core(clraddr, pinMask, ptr, numBytes);
        __enable_irq();
    } else {
        neopixel_send_buffer_core(clraddr, pinMask, ptr, numBytes);
    }
}

void neopixel_set_color(int argc, char** argv) {
	
	int color = atoi(argv[1]);
	
	
	PINOP(18, DIRSET);
    PINOP(18, OUTSET);
	
	LOG_W("NEOPIXEL", "color: %d", color);

    uint8_t buf[3];

    for (int i = 0; i <  3; i += 3) {
        buf[i + 0] = color >> 3;
        buf[i + 1] = color >> 2;
        buf[i + 2] = color >> 1;
    }
    neopixel_send_buffer(19, buf, 30);
}

SH_CMD_EXPORT(neopixel, neopixel_set_color, "XXX");
