#include "dashboard_ui.h"
#include <math.h> // For cos/sin in gauge needle updates

/* =======================================================
 *  Helpers (pure LVGL; no driver calls here)
 * ======================================================= */
static inline lv_color_t color_from_hex(uint32_t hex)
{
    return lv_color_make((hex >> 16) & 0xFF, (hex >> 8) & 0xFF, hex & 0xFF);
}

/* =======================================================
 * ===============  Super Mario  =========================
 * ======================================================= */
static void draw_box(lv_obj_t *parent, int x, int y, int w, int h, lv_color_t color, bool filled)
{
    lv_obj_t *box = lv_obj_create(parent);
    lv_obj_set_size(box, w, h);
    lv_obj_set_style_radius(box, 0, 0);
    lv_obj_set_pos(box, x, y);

    if (filled)
    {
        lv_obj_set_style_bg_color(box, color, 0);
        lv_obj_set_style_bg_opa(box, LV_OPA_COVER, 0);
        lv_obj_set_style_border_width(box, 0, 0);
        lv_obj_set_style_pad_all(box, 0, 0);
    }
    else
    {
        lv_obj_set_style_bg_opa(box, LV_OPA_TRANSP, 0);
        lv_obj_set_style_border_color(box, color, 0);
        lv_obj_set_style_border_width(box, 1, 0);
        lv_obj_set_style_border_side(box, LV_BORDER_SIDE_FULL, 0);
        lv_obj_set_style_pad_all(box, 0, 0);
    }
}

static void draw_circle(lv_obj_t *parent, int x, int y, int r, lv_color_t color)
{
    lv_obj_t *circle = lv_obj_create(parent);
    int size = 2 * r;
    lv_obj_set_size(circle, size, size);
    lv_obj_set_style_radius(circle, LV_RADIUS_CIRCLE, 0);
    lv_obj_set_style_bg_color(circle, color, 0);
    lv_obj_set_style_border_width(circle, 0, 0);
    lv_obj_set_pos(circle, x - r, y - r);
}

static void draw_line(lv_obj_t *parent, int x1, int y1, int x2, int y2, lv_color_t color)
{
    lv_obj_t *line_obj = lv_line_create(parent);
    static lv_point_precise_t line_points[2];
    line_points[0].x = x1;
    line_points[0].y = y1;
    line_points[1].x = x2;
    line_points[1].y = y2;

    lv_line_set_points(line_obj, line_points, 2);
    lv_obj_set_style_line_color(line_obj, color, 0);
    lv_obj_set_style_line_width(line_obj, 1, 0);
    lv_obj_set_style_bg_opa(line_obj, LV_OPA_TRANSP, 0);
}

static void draw_bitmap_placeholder(lv_obj_t *parent, int x, int y, int w, int h, lv_color_t color)
{
    draw_box(parent, x, y, w, h, color, false);
    if (w == 22 && h == 69)
    {
        draw_box(parent, x, y, w, h, color_from_hex(EPAPER_RED), true);
    }
}

static void draw_mario_scene(lv_obj_t *screen)
{
    lv_color_t white = color_from_hex(EPAPER_WHITE);
    lv_color_t black = color_from_hex(EPAPER_BLACK);
    lv_color_t red = color_from_hex(EPAPER_RED);
    lv_color_t green = color_from_hex(EPAPER_GREEN);
    lv_color_t yellow = color_from_hex(EPAPER_YELLOW);

    lv_obj_set_style_bg_color(screen, black, 0);
    lv_obj_set_style_bg_opa(screen, LV_OPA_COVER, 0);

    lv_obj_t *label_game = lv_label_create(screen);
    lv_label_set_text(label_game, "1 PLAYER GAME");
    lv_obj_set_style_text_color(label_game, white, 0);
    lv_obj_set_style_text_font(label_game, &lv_font_montserrat_24, 0);
    lv_obj_set_pos(label_game, 275, 169);

    lv_obj_t *label_super = lv_label_create(screen);
    lv_label_set_text(label_super, "super");
    lv_obj_set_style_text_color(label_super, green, 0);
    lv_obj_set_style_text_font(label_super, &lv_font_montserrat_32, 0);
    lv_obj_set_pos(label_super, 298, 11);

    lv_obj_t *label_mario = lv_label_create(screen);
    lv_label_set_text(label_mario, "Mario");
    lv_obj_set_style_text_color(label_mario, green, 0);
    lv_obj_set_style_text_font(label_mario, &lv_font_montserrat_32, 0);
    lv_obj_set_pos(label_mario, 298, 68);

    lv_obj_t *label_q = lv_label_create(screen);
    lv_label_set_text(label_q, "?");
    lv_obj_set_style_text_color(label_q, red, 0);
    lv_obj_set_style_text_font(label_q, &lv_font_montserrat_32, 0);
    lv_obj_set_pos(label_q, 451, 285);

    // Ground / bricks / lines / flagpole / coins / clouds… (full details preserved)
    int ground_y = 439, ground_h = 33, ground_w = 37;
    draw_box(screen, 3, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 39, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 76, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 113, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 149, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 185, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 222, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 259, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 295, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 331, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 368, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 405, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 442, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 478, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 515, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 552, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 587, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 623, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 660, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 697, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 733, ground_y, ground_w, ground_h, white, false);
    draw_box(screen, 770, ground_y, ground_w, ground_h, white, false);

    int stair_w = 37, stair_h = 32;
    draw_box(screen, 2, 407, stair_w, stair_h, white, false);
    draw_box(screen, 39, 407, stair_w, stair_h, white, false);
    draw_box(screen, 76, 407, stair_w, stair_h, white, false);

    draw_box(screen, 113, 254, stair_w, stair_h, white, false);
    draw_box(screen, 113, 283, stair_w, stair_h, white, false);
    draw_box(screen, 113, 314, stair_w, stair_h, white, false);
    draw_box(screen, 113, 346, stair_w, stair_h, white, false);
    draw_box(screen, 113, 377, stair_w, stair_h, white, false);
    draw_box(screen, 76, 315, stair_w, stair_h, white, false);
    draw_box(screen, 76, 345, stair_w, stair_h, white, false);
    draw_box(screen, 76, 376, stair_w, stair_h, white, false);
    draw_box(screen, 39, 346, stair_w, stair_h, white, false);
    draw_box(screen, 39, 377, stair_w, stair_h, white, false);
    draw_box(screen, 150, 315, stair_w, stair_h, white, false);
    draw_box(screen, 149, 346, stair_w, stair_h, white, false);
    draw_box(screen, 149, 378, stair_w, stair_h, white, false);
    draw_box(screen, 149, 408, stair_w, stair_h, white, false);
    draw_box(screen, 150, 224, stair_w, stair_h, white, false);
    draw_box(screen, 150, 254, stair_w, stair_h, white, false);
    draw_box(screen, 150, 284, stair_w, stair_h, white, false);
    draw_box(screen, 186, 348, stair_w, stair_h, white, false);
    draw_box(screen, 186, 378, stair_w, stair_h, white, false);
    draw_box(screen, 186, 408, stair_w, stair_h, white, false);
    draw_box(screen, 187, 256, stair_w, stair_h, white, false);
    draw_box(screen, 187, 286, stair_w, stair_h, white, false);
    draw_box(screen, 187, 316, stair_w, stair_h, white, false);
    draw_box(screen, 187, 165, stair_w, stair_h, white, false);
    draw_box(screen, 187, 195, stair_w, stair_h, white, false);
    draw_box(screen, 187, 225, stair_w, stair_h, white, false);
    draw_box(screen, 223, 348, stair_w, stair_h, white, false);
    draw_box(screen, 223, 378, stair_w, stair_h, white, false);
    draw_box(screen, 223, 408, stair_w, stair_h, white, false);
    draw_box(screen, 223, 256, stair_w, stair_h, white, false);
    draw_box(screen, 223, 286, stair_w, stair_h, white, false);
    draw_box(screen, 223, 316, stair_w, stair_h, white, false);
    draw_box(screen, 224, 165, stair_w, stair_h, white, false);
    draw_box(screen, 224, 195, stair_w, stair_h, white, false);
    draw_box(screen, 224, 225, stair_w, stair_h, white, false);

    draw_box(screen, 441, 280, 40, 34, white, false);

    draw_box(screen, 551, 408, stair_w, stair_h, white, false);
    draw_box(screen, 659, 345, stair_w, stair_h, white, false);
    draw_box(screen, 659, 375, stair_w, stair_h, white, false);
    draw_box(screen, 659, 405, stair_w, stair_h, white, false);
    draw_box(screen, 733, 346, stair_w, stair_h, white, false);
    draw_box(screen, 733, 376, stair_w, stair_h, white, false);
    draw_box(screen, 733, 406, stair_w, stair_h, white, false);
    draw_box(screen, 658, 313, stair_w, stair_h, white, false);
    draw_box(screen, 734, 313, stair_w, stair_h, white, false);
    draw_box(screen, 695, 281, stair_w, stair_h, white, false);

    draw_line(screen, 517, 161, 565, 161, white);
    draw_line(screen, 517, 161, 566, 199, white);

    draw_box(screen, 565, 153, 8, 255, white, true);

    draw_box(screen, 467, 352, 8, 8, white, true);
    draw_box(screen, 467, 344, 8, 8, white, true);
    draw_box(screen, 475, 359, 8, 8, white, true);
    draw_box(screen, 467, 367, 8, 8, white, true);
    draw_box(screen, 475, 367, 8, 8, white, true);
    draw_box(screen, 483, 367, 8, 8, white, true);
    draw_box(screen, 426, 344, 8, 8, white, true);
    draw_box(screen, 434, 344, 8, 8, white, true);
    draw_box(screen, 441, 344, 8, 8, white, true);
    draw_box(screen, 433, 352, 8, 8, white, true);
    draw_box(screen, 433, 358, 8, 8, white, true);
    draw_box(screen, 441, 358, 8, 8, white, true);
    draw_box(screen, 418, 353, 8, 8, white, true);
    draw_box(screen, 418, 361, 8, 8, white, true);

    draw_box(screen, 447, 330, 7, 7, red, true);
    draw_box(screen, 440, 330, 7, 7, red, true);
    draw_box(screen, 454, 330, 7, 7, red, true);
    draw_box(screen, 461, 330, 7, 7, red, true);
    draw_box(screen, 479, 337, 7, 7, red, true);
    draw_box(screen, 439, 337, 7, 7, red, true);
    draw_box(screen, 432, 337, 7, 7, red, true);
    draw_box(screen, 446, 337, 7, 7, red, true);
    draw_box(screen, 453, 337, 7, 7, red, true);
    draw_box(screen, 472, 337, 7, 7, red, true);
    draw_box(screen, 466, 337, 7, 7, red, true);
    draw_box(screen, 459, 337, 7, 7, red, true);

    draw_box(screen, 440, 377, 7, 7, red, true);
    draw_box(screen, 440, 383, 7, 7, red, true);
    draw_box(screen, 446, 385, 7, 7, red, true);
    draw_box(screen, 453, 385, 7, 7, red, true);
    draw_box(screen, 460, 385, 7, 7, red, true);
    draw_box(screen, 466, 382, 7, 7, red, true);
    draw_box(screen, 446, 391, 7, 7, red, true);
    draw_box(screen, 453, 391, 7, 7, red, true);
    draw_box(screen, 460, 391, 7, 7, red, true);
    draw_box(screen, 446, 397, 7, 7, red, true);
    draw_box(screen, 452, 397, 7, 7, red, true);
    draw_box(screen, 458, 397, 7, 7, red, true);
    draw_box(screen, 434, 393, 7, 7, red, true);
    draw_box(screen, 434, 400, 7, 7, red, true);
    draw_box(screen, 440, 400, 7, 7, red, true);
    draw_box(screen, 472, 393, 7, 7, red, true);
    draw_box(screen, 472, 400, 7, 7, red, true);
    draw_box(screen, 465, 400, 7, 7, red, true);
    draw_box(screen, 425, 406, 7, 7, red, true);
    draw_box(screen, 432, 406, 7, 7, red, true);
    draw_box(screen, 439, 406, 7, 7, red, true);
    draw_box(screen, 425, 412, 7, 7, red, true);
    draw_box(screen, 432, 412, 7, 7, red, true);
    draw_box(screen, 439, 412, 7, 7, red, true);
    draw_box(screen, 425, 418, 7, 7, red, true);
    draw_box(screen, 431, 418, 7, 7, red, true);
    draw_box(screen, 437, 418, 7, 7, red, true);
    draw_box(screen, 469, 407, 7, 7, red, true);
    draw_box(screen, 476, 407, 7, 7, red, true);
    draw_box(screen, 483, 407, 7, 7, red, true);
    draw_box(screen, 469, 413, 7, 7, red, true);
    draw_box(screen, 476, 413, 7, 7, red, true);
    draw_box(screen, 483, 413, 7, 7, red, true);
    draw_box(screen, 469, 419, 7, 7, red, true);
    draw_box(screen, 475, 419, 7, 7, red, true);
    draw_box(screen, 481, 419, 7, 7, red, true);

    draw_box(screen, 430, 424, 8, 8, white, true);
    draw_box(screen, 422, 424, 8, 8, white, true);
    draw_box(screen, 415, 426, 8, 8, white, true);
    draw_box(screen, 409, 431, 8, 8, white, true);
    draw_box(screen, 417, 431, 8, 8, white, true);
    draw_box(screen, 425, 432, 8, 8, white, true);
    draw_box(screen, 431, 432, 8, 8, white, true);
    draw_box(screen, 480, 426, 8, 8, white, true);
    draw_box(screen, 472, 426, 8, 8, white, true);
    draw_box(screen, 485, 427, 8, 8, white, true);
    draw_box(screen, 484, 432, 8, 8, white, true);
    draw_box(screen, 472, 432, 8, 8, white, true);
    draw_box(screen, 478, 432, 8, 8, white, true);
    draw_box(screen, 491, 431, 8, 8, white, true);
    draw_box(screen, 418, 368, 8, 8, white, true);
    draw_box(screen, 426, 368, 8, 8, white, true);
    draw_box(screen, 432, 377, 8, 8, white, true);
    draw_box(screen, 432, 385, 8, 8, white, true);
    draw_box(screen, 447, 377, 8, 8, white, true);
    draw_box(screen, 408, 387, 8, 8, white, true);
    draw_box(screen, 466, 375, 6, 7, white, true);
    draw_box(screen, 455, 377, 8, 8, white, true);
    draw_box(screen, 421, 398, 8, 8, white, true);
    draw_box(screen, 426, 398, 8, 8, white, true);
    draw_box(screen, 416, 387, 8, 8, white, true);
    draw_box(screen, 408, 395, 8, 8, white, true);
    draw_box(screen, 416, 394, 8, 8, white, true);
    draw_box(screen, 408, 402, 8, 8, white, true);
    draw_box(screen, 415, 402, 8, 8, white, true);
    draw_box(screen, 486, 386, 8, 8, white, true);
    draw_box(screen, 479, 396, 8, 8, white, true);
    draw_box(screen, 494, 386, 8, 8, white, true);
    draw_box(screen, 486, 392, 8, 8, white, true);
    draw_box(screen, 494, 393, 8, 8, white, true);
    draw_box(screen, 486, 399, 8, 8, white, true);
    draw_box(screen, 494, 399, 8, 8, white, true);

    draw_circle(screen, 568, 146, 9, green);
    draw_circle(screen, 32, 92, 11, white);
    draw_circle(screen, 43, 82, 11, white);
    draw_circle(screen, 53, 91, 11, white);
    draw_circle(screen, 183, 59, 11, white);
    draw_circle(screen, 172, 69, 11, white);
    draw_circle(screen, 193, 68, 11, white);
    draw_circle(screen, 641, 99, 11, white);
    draw_circle(screen, 626, 101, 11, white);
    draw_circle(screen, 651, 108, 11, white);
    draw_circle(screen, 45, 84, 11, white);
    draw_circle(screen, 34, 94, 11, white);
    draw_circle(screen, 55, 93, 11, white);
    draw_circle(screen, 605, 95, 11, white);
    draw_circle(screen, 594, 105, 11, white);
    draw_circle(screen, 615, 104, 11, white);
    draw_circle(screen, 789, 80, 11, white);
    draw_circle(screen, 778, 90, 11, white);
    draw_circle(screen, 799, 89, 11, white);

    draw_circle(screen, 459, 237, 13, yellow);
    draw_circle(screen, 519, 275, 13, yellow);
    draw_circle(screen, 412, 264, 13, yellow);

    draw_box(screen, 478, 173, 49, 1, white, false);
    draw_box(screen, 320, 367, 22, 69, white, false);

    draw_box(screen, 314, 389, 34, 51, green, true);
    draw_box(screen, 298, 365, 64, 24, green, true);
    draw_box(screen, 314, 389, 34, 51, white, false);
    draw_box(screen, 298, 365, 64, 24, white, false);
}

void create_supermario_ui()
{
    lv_obj_t *screen = lv_scr_act();
    draw_mario_scene(screen);
}

/* =======================================================
 * ===============  Smart Home  ==========================
 * ======================================================= */
/* Label pointers stored for runtime updates */
static lv_obj_t *smh_temp_label;
static lv_obj_t *smh_humidity_label;
static lv_obj_t *smh_battery_voltage_label;
static lv_obj_t *smh_battery_capacity_label;
static lv_obj_t *smh_wifi_signal_label;
static lv_obj_t *smh_weather_status_label;
static lv_obj_t *smh_location_label;
static lv_obj_t *smh_email_icon;
static lv_obj_t *smh_seeed_studio_label;

void create_smarthome_ui()
{
    lv_obj_t *screen = lv_scr_act();
    lv_obj_set_style_bg_color(screen, color_from_hex(EPAPER_WHITE), 0);

    lv_obj_t *main_container = lv_obj_create(screen);
    lv_obj_set_size(main_container, lv_pct(100), lv_pct(100));
    lv_obj_set_pos(main_container, 0, 0);
    lv_obj_set_style_bg_color(main_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(main_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(main_container, BORDER_WIDTH, 0);
    lv_obj_set_style_pad_all(main_container, 5, 0);

    lv_obj_t *header = lv_obj_create(main_container);
    lv_obj_set_size(header, 790, 50);
    lv_obj_set_pos(header, 5, 5);
    lv_obj_set_style_bg_color(header, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_radius(header, CORNER_RADIUS, 0);
    lv_obj_set_style_border_width(header, 0, 0);

    lv_obj_t *main_title = lv_label_create(header);
    lv_label_set_text(main_title, "SMART HOME DASHBOARD");
    lv_obj_set_style_text_color(main_title, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_text_font(main_title, &lv_font_montserrat_20, 0);
    lv_obj_center(main_title);

    int content_y_start = 60;
    int screen_height = 480;
    int content_height = screen_height - content_y_start - 5;
    int gap = 10;
    int total_region_height = content_height - (2 * gap);
    int top_region_height = (total_region_height * 2) / 8;
    int center_region_height = (total_region_height * 2) / 8;
    int bottom_region_height = total_region_height - top_region_height - center_region_height;
    int top_y = content_y_start;
    int center_y = top_y + top_region_height + gap;
    int bottom_y = center_y + center_region_height + gap;

    lv_obj_t *top_left_region = lv_obj_create(main_container);
    lv_obj_set_size(top_left_region, 390, top_region_height);
    lv_obj_set_pos(top_left_region, 5, top_y);
    lv_obj_set_style_bg_color(top_left_region, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(top_left_region, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(top_left_region, BORDER_WIDTH, 0);
    lv_obj_set_style_radius(top_left_region, CORNER_RADIUS, 0);
    lv_obj_set_style_pad_all(top_left_region, PANEL_PADDING, 0);

    lv_obj_t *top_right_region = lv_obj_create(main_container);
    lv_obj_set_size(top_right_region, 390, top_region_height);
    lv_obj_set_pos(top_right_region, 405, top_y);
    lv_obj_set_style_bg_color(top_right_region, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(top_right_region, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(top_right_region, BORDER_WIDTH, 0);
    lv_obj_set_style_radius(top_right_region, CORNER_RADIUS, 0);
    lv_obj_set_style_pad_all(top_right_region, PANEL_PADDING, 0);

    lv_obj_t *center_region = lv_obj_create(main_container);
    lv_obj_set_size(center_region, 790, center_region_height);
    lv_obj_set_pos(center_region, 5, center_y);
    lv_obj_set_style_bg_color(center_region, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(center_region, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(center_region, BORDER_WIDTH, 0);
    lv_obj_set_style_radius(center_region, CORNER_RADIUS, 0);
    lv_obj_set_style_pad_all(center_region, PANEL_PADDING, 0);

    lv_obj_t *bottom_left_region = lv_obj_create(main_container);
    lv_obj_set_size(bottom_left_region, 390, bottom_region_height);
    lv_obj_set_pos(bottom_left_region, 5, bottom_y);
    lv_obj_set_style_bg_color(bottom_left_region, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(bottom_left_region, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(bottom_left_region, BORDER_WIDTH, 0);
    lv_obj_set_style_radius(bottom_left_region, CORNER_RADIUS, 0);
    lv_obj_set_style_pad_all(bottom_left_region, PANEL_PADDING, 0);

    lv_obj_t *bottom_right_region = lv_obj_create(main_container);
    lv_obj_set_size(bottom_right_region, 390, bottom_region_height);
    lv_obj_set_pos(bottom_right_region, 405, bottom_y);
    lv_obj_set_style_bg_color(bottom_right_region, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(bottom_right_region, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(bottom_right_region, BORDER_WIDTH, 0);
    lv_obj_set_style_radius(bottom_right_region, CORNER_RADIUS, 0);
    lv_obj_set_style_pad_all(bottom_right_region, PANEL_PADDING, 0);

    // Center: Humidity
    lv_obj_t *humidity_container = lv_obj_create(center_region);
    lv_obj_set_size(humidity_container, 350, center_region_height - 20);
    lv_obj_set_pos(humidity_container, 10, 10);
    lv_obj_set_style_bg_color(humidity_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(humidity_container, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_border_width(humidity_container, 2, 0);
    lv_obj_set_style_radius(humidity_container, CORNER_RADIUS, 0);

    lv_obj_t *humidity_title = lv_label_create(humidity_container);
    lv_label_set_text(humidity_title, "Humidity");
    lv_obj_set_style_text_color(humidity_title, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(humidity_title, &lv_font_montserrat_16, 0);
    lv_obj_align(humidity_title, LV_ALIGN_TOP_MID, 0, 5);

    lv_obj_t *humidity_icon_img = lv_label_create(humidity_container);
    lv_label_set_text(humidity_icon_img, LV_SYMBOL_TINT);
    lv_obj_set_style_text_font(humidity_icon_img, &lv_font_montserrat_28, 0);
    lv_obj_set_style_text_color(humidity_icon_img, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_align(humidity_icon_img, LV_ALIGN_LEFT_MID, 20, 10);

    smh_humidity_label = lv_label_create(humidity_container);
    lv_label_set_text(smh_humidity_label, "45%");
    lv_obj_set_style_text_color(smh_humidity_label, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(smh_humidity_label, &lv_font_montserrat_24, 0);
    lv_obj_align(smh_humidity_label, LV_ALIGN_RIGHT_MID, -40, 10);

    // Center: Temperature
    lv_obj_t *temp_container = lv_obj_create(center_region);
    lv_obj_set_size(temp_container, 350, center_region_height - 20);
    lv_obj_set_pos(temp_container, 420, 10);
    lv_obj_set_style_bg_color(temp_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(temp_container, color_from_hex(EPAPER_RED), 0);
    lv_obj_set_style_border_width(temp_container, 2, 0);
    lv_obj_set_style_radius(temp_container, CORNER_RADIUS, 0);

    lv_obj_t *temp_title = lv_label_create(temp_container);
    lv_label_set_text(temp_title, "Temperature");
    lv_obj_set_style_text_color(temp_title, color_from_hex(EPAPER_RED), 0);
    lv_obj_set_style_text_font(temp_title, &lv_font_montserrat_16, 0);
    lv_obj_align(temp_title, LV_ALIGN_TOP_MID, 0, 5);

    lv_obj_t *temp_icon_img = lv_label_create(temp_container);
    lv_label_set_text(temp_icon_img, "°");
    lv_obj_set_style_text_font(temp_icon_img, &lv_font_montserrat_28, 0);
    lv_obj_set_style_text_color(temp_icon_img, color_from_hex(EPAPER_RED), 0);
    lv_obj_align(temp_icon_img, LV_ALIGN_LEFT_MID, 20, 10);

    smh_temp_label = lv_label_create(temp_container);
    lv_label_set_text(smh_temp_label, "22°C");
    lv_obj_set_style_text_color(smh_temp_label, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(smh_temp_label, &lv_font_montserrat_24, 0);
    lv_obj_align(smh_temp_label, LV_ALIGN_RIGHT_MID, -40, 10);

    // Top right: Battery & WiFi
    lv_obj_t *battery_title = lv_label_create(top_right_region);
    lv_label_set_text(battery_title, "BATTERY & NETWORK");
    lv_obj_set_style_text_color(battery_title, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(battery_title, &lv_font_montserrat_16, 0);
    lv_obj_align(battery_title, LV_ALIGN_TOP_LEFT, 0, 5);

    lv_obj_t *voltage_title = lv_label_create(top_right_region);
    lv_label_set_text(voltage_title, "Voltage:");
    lv_obj_set_style_text_font(voltage_title, &lv_font_montserrat_14, 0);
    lv_obj_align(voltage_title, LV_ALIGN_TOP_LEFT, 5, 30);

    smh_battery_voltage_label = lv_label_create(top_right_region);
    lv_label_set_text(smh_battery_voltage_label, "12.4V");
    lv_obj_set_style_text_font(smh_battery_voltage_label, &lv_font_montserrat_16, 0);
    lv_obj_align_to(smh_battery_voltage_label, voltage_title, LV_ALIGN_OUT_RIGHT_MID, 5, 0);

    lv_obj_t *capacity_title = lv_label_create(top_right_region);
    lv_label_set_text(capacity_title, "Capacity:");
    lv_obj_set_style_text_font(capacity_title, &lv_font_montserrat_14, 0);
    lv_obj_align_to(capacity_title, smh_battery_voltage_label, LV_ALIGN_OUT_RIGHT_MID, 20, 0);

    smh_battery_capacity_label = lv_label_create(top_right_region);
    lv_label_set_text(smh_battery_capacity_label, "85%");
    lv_obj_set_style_text_font(smh_battery_capacity_label, &lv_font_montserrat_16, 0);
    lv_obj_align_to(smh_battery_capacity_label, capacity_title, LV_ALIGN_OUT_RIGHT_MID, 5, 0);

    lv_obj_t *wifi_title = lv_label_create(top_right_region);
    lv_label_set_text(wifi_title, "WiFi:");
    lv_obj_set_style_text_font(wifi_title, &lv_font_montserrat_14, 0);
    lv_obj_align(wifi_title, LV_ALIGN_TOP_LEFT, 5, 60);

    smh_wifi_signal_label = lv_label_create(top_right_region);
    lv_label_set_text(smh_wifi_signal_label, "■■■■□");
    lv_obj_set_style_text_color(smh_wifi_signal_label, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(smh_wifi_signal_label, &lv_font_montserrat_16, 0);
    lv_obj_align_to(smh_wifi_signal_label, wifi_title, LV_ALIGN_OUT_RIGHT_MID, 10, 0);

    // Top left: Weather & Location
    lv_obj_t *weather_title = lv_label_create(top_left_region);
    lv_label_set_text(weather_title, "WEATHER");
    lv_obj_set_style_text_color(weather_title, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(weather_title, &lv_font_montserrat_16, 0);
    lv_obj_align(weather_title, LV_ALIGN_TOP_LEFT, 0, 5);

    smh_weather_status_label = lv_label_create(top_left_region);
    lv_label_set_text(smh_weather_status_label, "Sunny");
    lv_obj_set_style_text_font(smh_weather_status_label, &lv_font_montserrat_16, 0);
    lv_obj_align(smh_weather_status_label, LV_ALIGN_TOP_LEFT, 5, 30);

    lv_obj_t *location_title = lv_label_create(top_left_region);
    lv_label_set_text(location_title, "Location:");
    lv_obj_set_style_text_font(location_title, &lv_font_montserrat_14, 0);
    lv_obj_align(location_title, LV_ALIGN_TOP_LEFT, 5, 60);

    smh_location_label = lv_label_create(top_left_region);
    lv_label_set_text(smh_location_label, "Beijing");
    lv_obj_set_style_text_font(smh_location_label, &lv_font_montserrat_16, 0);
    lv_obj_align_to(smh_location_label, location_title, LV_ALIGN_OUT_RIGHT_MID, 10, 0);

    // Bottom left: To-do
    lv_obj_t *todo_title = lv_label_create(bottom_left_region);
    lv_label_set_text(todo_title, "TO-DO LIST");
    lv_obj_set_style_text_color(todo_title, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(todo_title, &lv_font_montserrat_16, 0);
    lv_obj_align(todo_title, LV_ALIGN_TOP_LEFT, 0, 5);

    lv_obj_t *cb1 = lv_checkbox_create(bottom_left_region);
    lv_checkbox_set_text(cb1, "Water plants");
    lv_obj_align(cb1, LV_ALIGN_TOP_LEFT, 5, 35);

    lv_obj_t *cb2 = lv_checkbox_create(bottom_left_region);
    lv_checkbox_set_text(cb2, "Check security");
    lv_obj_align_to(cb2, cb1, LV_ALIGN_OUT_BOTTOM_LEFT, 0, 5);

    lv_obj_t *cb3 = lv_checkbox_create(bottom_left_region);
    lv_checkbox_set_text(cb3, "Update firmware");
    lv_obj_align_to(cb3, cb2, LV_ALIGN_OUT_BOTTOM_LEFT, 0, 5);

    lv_obj_t *smiley_face = lv_label_create(bottom_left_region);
    lv_label_set_text(smiley_face, LV_SYMBOL_OK);
    lv_obj_set_style_text_font(smiley_face, &lv_font_montserrat_28, 0);
    lv_obj_set_style_text_color(smiley_face, color_from_hex(EPAPER_YELLOW), 0);
    lv_obj_align(smiley_face, LV_ALIGN_BOTTOM_RIGHT, -15, -15);

    // Bottom right: Contact
    lv_obj_t *email_title = lv_label_create(bottom_right_region);
    lv_label_set_text(email_title, "CONTACT");
    lv_obj_set_style_text_color(email_title, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(email_title, &lv_font_montserrat_16, 0);
    lv_obj_align(email_title, LV_ALIGN_TOP_LEFT, 0, 5);

    smh_email_icon = lv_label_create(bottom_right_region);
    lv_label_set_text(smh_email_icon, "Email");
    lv_obj_set_style_text_font(smh_email_icon, &lv_font_montserrat_28, 0);
    lv_obj_align(smh_email_icon, LV_ALIGN_CENTER, 0, -15);

    smh_seeed_studio_label = lv_label_create(bottom_right_region);
    lv_label_set_text(smh_seeed_studio_label, "Seeed Studio");
    lv_obj_set_style_text_color(smh_seeed_studio_label, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(smh_seeed_studio_label, &lv_font_montserrat_18, 0);
    lv_obj_align(smh_seeed_studio_label, LV_ALIGN_CENTER, 0, 20);
}

/* Smart Home updates */
void update_temperature(float temp)
{
    static char buf[32];
    lv_snprintf(buf, sizeof(buf), "%.1f°C", temp);
    lv_label_set_text(smh_temp_label, buf);
    // Optional visual cue
    lv_color_t color = (temp > 25) ? color_from_hex(EPAPER_RED) : (temp < 18) ? color_from_hex(EPAPER_BLUE)
                                                                              : color_from_hex(EPAPER_BLACK);
    lv_obj_set_style_text_color(smh_temp_label, color, 0);
}

void update_humidity(int humidity)
{
    static char buf[32];
    lv_snprintf(buf, sizeof(buf), "%d%%", humidity);
    lv_label_set_text(smh_humidity_label, buf);
    lv_color_t color = (humidity > 70) ? color_from_hex(EPAPER_BLUE) : (humidity < 30) ? color_from_hex(EPAPER_RED)
                                                                                       : color_from_hex(EPAPER_BLACK);
    lv_obj_set_style_text_color(smh_humidity_label, color, 0);
}

void update_battery_voltage(float voltage)
{
    static char buf[32];
    lv_snprintf(buf, sizeof(buf), "%.1fV", voltage);
    lv_label_set_text(smh_battery_voltage_label, buf);
    lv_color_t color = (voltage < 11.0) ? color_from_hex(EPAPER_RED) : (voltage > 14.0) ? color_from_hex(EPAPER_YELLOW)
                                                                                        : color_from_hex(EPAPER_BLACK);
    lv_obj_set_style_text_color(smh_battery_voltage_label, color, 0);
}

void update_battery_capacity(int percentage)
{
    if (percentage < 0)
        percentage = 0;
    if (percentage > 100)
        percentage = 100;
    static char buf[32];
    lv_snprintf(buf, sizeof(buf), "%d%%", percentage);
    lv_label_set_text(smh_battery_capacity_label, buf);
    lv_color_t color = (percentage < 20) ? color_from_hex(EPAPER_RED) : (percentage < 50) ? color_from_hex(EPAPER_YELLOW)
                                                                                          : color_from_hex(EPAPER_GREEN);
    lv_obj_set_style_text_color(smh_battery_capacity_label, color, 0);
}

void update_wifi_signal(int signal)
{
    const char *signal_text[] = {"□□□□", "■□□□", "■■□□", "■■■□", "■■■■"};
    if (signal < 0)
        signal = 0;
    if (signal > 4)
        signal = 4;
    lv_label_set_text(smh_wifi_signal_label, signal_text[signal]);
    lv_color_t color = (signal < 2) ? color_from_hex(EPAPER_RED) : (signal < 4) ? color_from_hex(EPAPER_YELLOW)
                                                                                : color_from_hex(EPAPER_GREEN);
    lv_obj_set_style_text_color(smh_wifi_signal_label, color, 0);
}

void update_weather_status(const char *status)
{
    lv_label_set_text(smh_weather_status_label, status);
}

void update_location(const char *location)
{
    lv_label_set_text(smh_location_label, location);
}

/* For compatibility with older code (no UI list) */
void add_todo_item(const char *item) { (void)item; }
void remove_todo_item(int) {}

/* =======================================================
 * ===============  Vehicle Dashboard  ===================
 * ======================================================= */
static lv_obj_t *veh_speed_gauge;
static lv_obj_t *veh_rpm_gauge;
static lv_obj_t *veh_speed_label;
static lv_obj_t *veh_rpm_label;
static lv_obj_t *veh_speed_unit_label;
static lv_obj_t *veh_rpm_unit_label;
static lv_obj_t *veh_fuel_label;
static lv_obj_t *veh_temp_label;
static lv_obj_t *veh_gear_label;
static lv_obj_t *veh_odometer_label;
static lv_obj_t *veh_warning_panel;
static lv_obj_t *veh_info_panel;

void create_vehicle_dashboard_ui()
{
    lv_obj_t *screen = lv_scr_act();
    lv_obj_set_style_bg_color(screen, color_from_hex(EPAPER_WHITE), 0);

    lv_obj_t *main_container = lv_obj_create(screen);
    lv_obj_set_size(main_container, lv_pct(100), lv_pct(100));
    lv_obj_set_style_bg_color(main_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(main_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(main_container, BORDER_WIDTH, 0);
    lv_obj_set_style_pad_all(main_container, PANEL_PADDING, 0);

    lv_obj_t *header = lv_obj_create(main_container);
    lv_obj_set_size(header, lv_pct(100), 50);
    lv_obj_set_style_bg_color(header, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_radius(header, CORNER_RADIUS, 0);

    lv_obj_t *main_title = lv_label_create(header);
    lv_label_set_text(main_title, "VEHICLE DASHBOARD");
    lv_obj_set_style_text_color(main_title, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_text_font(main_title, &lv_font_montserrat_20, 0);
    lv_obj_center(main_title);

    lv_obj_t *gauge_panel = lv_obj_create(main_container);
    lv_obj_set_size(gauge_panel, lv_pct(100), 300);
    lv_obj_set_style_bg_color(gauge_panel, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(gauge_panel, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(gauge_panel, BORDER_WIDTH, 0);
    lv_obj_set_style_radius(gauge_panel, CORNER_RADIUS, 0);
    lv_obj_align_to(gauge_panel, header, LV_ALIGN_OUT_BOTTOM_MID, 0, PANEL_PADDING);

    // Speed gauge container
    lv_obj_t *speed_gauge_container = lv_obj_create(gauge_panel);
    lv_obj_set_size(speed_gauge_container, LARGE_GAUGE_SIZE + 50, LARGE_GAUGE_SIZE + 60);
    lv_obj_set_pos(speed_gauge_container, 125 - (LARGE_GAUGE_SIZE + 50) / 2,
                   130 - (LARGE_GAUGE_SIZE + 60) / 2);
    lv_obj_set_style_bg_color(speed_gauge_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(speed_gauge_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(speed_gauge_container, 2, 0);
    lv_obj_set_style_radius(speed_gauge_container, 15, 0);

    lv_obj_t *speed_title = lv_label_create(speed_gauge_container);
    lv_label_set_text(speed_title, "SPEED");
    lv_obj_set_style_text_color(speed_title, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(speed_title, &lv_font_montserrat_18, 0);
    lv_obj_align(speed_title, LV_ALIGN_TOP_MID, 0, -10);

    lv_obj_t *speed_bg = lv_obj_create(speed_gauge_container);
    lv_obj_set_size(speed_bg, LARGE_GAUGE_SIZE, LARGE_GAUGE_SIZE);
    lv_obj_center(speed_bg);
    lv_obj_set_style_bg_color(speed_bg, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(speed_bg, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_border_width(speed_bg, 3, 0);
    lv_obj_set_style_radius(speed_bg, LARGE_GAUGE_SIZE / 2, 0);

    // RPM gauge container
    lv_obj_t *rpm_gauge_container = lv_obj_create(gauge_panel);
    lv_obj_set_size(rpm_gauge_container, LARGE_GAUGE_SIZE + 50, LARGE_GAUGE_SIZE + 60);
    lv_obj_set_pos(rpm_gauge_container, 600 - (LARGE_GAUGE_SIZE + 50) / 2,
                   130 - (LARGE_GAUGE_SIZE + 60) / 2);
    lv_obj_set_style_bg_color(rpm_gauge_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(rpm_gauge_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(rpm_gauge_container, 2, 0);
    lv_obj_set_style_radius(rpm_gauge_container, 15, 0);

    lv_obj_t *rpm_title = lv_label_create(rpm_gauge_container);
    lv_label_set_text(rpm_title, "ENGINE RPM");
    lv_obj_set_style_text_color(rpm_title, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(rpm_title, &lv_font_montserrat_18, 0);
    lv_obj_align(rpm_title, LV_ALIGN_TOP_MID, 0, -10);

    lv_obj_t *rpm_bg = lv_obj_create(rpm_gauge_container);
    lv_obj_set_size(rpm_bg, LARGE_GAUGE_SIZE, LARGE_GAUGE_SIZE);
    lv_obj_center(rpm_bg);
    lv_obj_set_style_bg_color(rpm_bg, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(rpm_bg, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(rpm_bg, 3, 0);
    lv_obj_set_style_radius(rpm_bg, LARGE_GAUGE_SIZE / 2, 0);

    // Needles (static geometry; points updated in update_* functions)
    lv_obj_t *speed_needle = lv_line_create(speed_bg);
    static lv_point_precise_t speed_needle_points[2];
    speed_needle_points[0].x = LARGE_GAUGE_SIZE / 2;
    speed_needle_points[0].y = LARGE_GAUGE_SIZE / 2;
    speed_needle_points[1].x = LARGE_GAUGE_SIZE / 2;
    speed_needle_points[1].y = LARGE_GAUGE_SIZE / 2 - ((LARGE_GAUGE_SIZE / 2 - 3) - 10);
    lv_line_set_points(speed_needle, speed_needle_points, 2);
    lv_obj_set_style_line_color(speed_needle, color_from_hex(EPAPER_RED), 0);
    lv_obj_set_style_line_width(speed_needle, 5, 0);

    lv_obj_t *speed_center_dot = lv_obj_create(speed_bg);
    lv_obj_set_size(speed_center_dot, 16, 16);
    lv_obj_set_pos(speed_center_dot, LARGE_GAUGE_SIZE / 2 - 8, LARGE_GAUGE_SIZE / 2 - 8);
    lv_obj_set_style_bg_color(speed_center_dot, color_from_hex(EPAPER_RED), 0);
    lv_obj_set_style_border_width(speed_center_dot, 0, 0);
    lv_obj_set_style_radius(speed_center_dot, 8, 0);

    lv_obj_t *rpm_needle = lv_line_create(rpm_bg);
    static lv_point_precise_t rpm_needle_points[2];
    rpm_needle_points[0].x = LARGE_GAUGE_SIZE / 2;
    rpm_needle_points[0].y = LARGE_GAUGE_SIZE / 2;
    rpm_needle_points[1].x = LARGE_GAUGE_SIZE / 2;
    rpm_needle_points[1].y = LARGE_GAUGE_SIZE / 2 - ((LARGE_GAUGE_SIZE / 2 - 3) - 10);
    lv_line_set_points(rpm_needle, rpm_needle_points, 2);
    lv_obj_set_style_line_color(rpm_needle, color_from_hex(EPAPER_RED), 0);
    lv_obj_set_style_line_width(rpm_needle, 5, 0);

    lv_obj_t *rpm_center_dot = lv_obj_create(rpm_bg);
    lv_obj_set_size(rpm_center_dot, 16, 16);
    lv_obj_set_pos(rpm_center_dot, LARGE_GAUGE_SIZE / 2 - 8, LARGE_GAUGE_SIZE / 2 - 8);
    lv_obj_set_style_bg_color(rpm_center_dot, color_from_hex(EPAPER_RED), 0);
    lv_obj_set_style_border_width(rpm_center_dot, 0, 0);
    lv_obj_set_style_radius(rpm_center_dot, 8, 0);

    // Digit readouts
    veh_speed_label = lv_label_create(speed_gauge_container);
    lv_label_set_text(veh_speed_label, "0");
    lv_obj_set_style_text_color(veh_speed_label, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(veh_speed_label, &lv_font_montserrat_28, 0);
    lv_obj_align(veh_speed_label, LV_ALIGN_CENTER, 0, 0);

    veh_speed_unit_label = lv_label_create(speed_gauge_container);
    lv_label_set_text(veh_speed_unit_label, "km/h");
    lv_obj_set_style_text_color(veh_speed_unit_label, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(veh_speed_unit_label, &lv_font_montserrat_18, 0);
    lv_obj_align(veh_speed_unit_label, LV_ALIGN_CENTER, 0, 30);

    veh_rpm_label = lv_label_create(rpm_gauge_container);
    lv_label_set_text(veh_rpm_label, "0");
    lv_obj_set_style_text_color(veh_rpm_label, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(veh_rpm_label, &lv_font_montserrat_28, 0);
    lv_obj_align(veh_rpm_label, LV_ALIGN_CENTER, 0, 0);

    veh_rpm_unit_label = lv_label_create(rpm_gauge_container);
    lv_label_set_text(veh_rpm_unit_label, "x1000 RPM");
    lv_obj_set_style_text_color(veh_rpm_unit_label, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(veh_rpm_unit_label, &lv_font_montserrat_18, 0);
    lv_obj_align(veh_rpm_unit_label, LV_ALIGN_CENTER, 0, 30);

    // Center info panel
    veh_info_panel = lv_obj_create(gauge_panel);
    lv_obj_set_size(veh_info_panel, 220, 150);
    lv_obj_set_style_bg_color(veh_info_panel, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_width(veh_info_panel, 0, 0);
    lv_obj_align(veh_info_panel, LV_ALIGN_CENTER, 0, 20);

    // Gear
    lv_obj_t *gear_container = lv_obj_create(veh_info_panel);
    lv_obj_set_size(gear_container, 80, 50);
    lv_obj_set_style_bg_color(gear_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_radius(gear_container, CORNER_RADIUS, 0);
    lv_obj_align(gear_container, LV_ALIGN_TOP_MID, 0, 5);

    veh_gear_label = lv_label_create(gear_container);
    lv_label_set_text(veh_gear_label, "N");
    lv_obj_set_style_text_color(veh_gear_label, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_text_font(veh_gear_label, &lv_font_montserrat_28, 0);
    lv_obj_center(veh_gear_label);

    // Fuel
    lv_obj_t *fuel_container = lv_obj_create(veh_info_panel);
    lv_obj_set_size(fuel_container, 90, 40);
    lv_obj_set_style_bg_color(fuel_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(fuel_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(fuel_container, 2, 0);
    lv_obj_align(fuel_container, LV_ALIGN_BOTTOM_LEFT, 5, -5);

    lv_obj_t *fuel_title = lv_label_create(fuel_container);
    lv_label_set_text(fuel_title, "FUEL");
    lv_obj_set_style_text_color(fuel_title, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(fuel_title, &lv_font_montserrat_12, 0);
    lv_obj_align(fuel_title, LV_ALIGN_TOP_MID, 0, 2);

    veh_fuel_label = lv_label_create(fuel_container);
    lv_label_set_text(veh_fuel_label, "100%");
    lv_obj_set_style_text_color(veh_fuel_label, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(veh_fuel_label, &lv_font_montserrat_16, 0);
    lv_obj_align(veh_fuel_label, LV_ALIGN_BOTTOM_MID, 0, -2);

    // Water temp
    lv_obj_t *temp_container2 = lv_obj_create(veh_info_panel);
    lv_obj_set_size(temp_container2, 90, 40);
    lv_obj_set_style_bg_color(temp_container2, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(temp_container2, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(temp_container2, 2, 0);
    lv_obj_align(temp_container2, LV_ALIGN_BOTTOM_RIGHT, -5, -5);

    lv_obj_t *temp_title2 = lv_label_create(temp_container2);
    lv_label_set_text(temp_title2, "TEMP");
    lv_obj_set_style_text_color(temp_title2, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(temp_title2, &lv_font_montserrat_12, 0);
    lv_obj_align(temp_title2, LV_ALIGN_TOP_MID, 0, 2);

    veh_temp_label = lv_label_create(temp_container2);
    lv_label_set_text(veh_temp_label, "25°C");
    lv_obj_set_style_text_color(veh_temp_label, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(veh_temp_label, &lv_font_montserrat_16, 0);
    lv_obj_align(veh_temp_label, LV_ALIGN_BOTTOM_MID, 0, -2);

    // Odometer
    lv_obj_t *odo_container = lv_obj_create(main_container);
    lv_obj_set_size(odo_container, 280, 40);
    lv_obj_set_style_bg_color(odo_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_radius(odo_container, CORNER_RADIUS, 0);
    lv_obj_align_to(odo_container, gauge_panel, LV_ALIGN_BOTTOM_MID, 0, 5);

    lv_obj_t *odo_title = lv_label_create(odo_container);
    lv_label_set_text(odo_title, "ODOMETER:");
    lv_obj_set_style_text_color(odo_title, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_text_font(odo_title, &lv_font_montserrat_16, 0);
    lv_obj_align(odo_title, LV_ALIGN_LEFT_MID, 10, 0);

    veh_odometer_label = lv_label_create(odo_container);
    lv_label_set_text(veh_odometer_label, "000000 km");
    lv_obj_set_style_text_color(veh_odometer_label, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_text_font(veh_odometer_label, &lv_font_montserrat_16, 0);
    lv_obj_align(veh_odometer_label, LV_ALIGN_RIGHT_MID, -10, 0);

    // Warning panel
    lv_obj_t *warning_panel_container = lv_obj_create(main_container);
    lv_obj_set_size(warning_panel_container, lv_pct(100), 80);
    lv_obj_set_style_bg_color(warning_panel_container, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(warning_panel_container, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(warning_panel_container, BORDER_WIDTH, 0);
    lv_obj_set_style_radius(warning_panel_container, CORNER_RADIUS, 0);
    lv_obj_align_to(warning_panel_container, gauge_panel, LV_ALIGN_OUT_BOTTOM_MID, 0, PANEL_PADDING);

    lv_obj_t *warning_title = lv_label_create(warning_panel_container);
    lv_label_set_text(warning_title, "VEHICLE STATUS");
    lv_obj_set_style_text_color(warning_title, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_text_font(warning_title, &lv_font_montserrat_16, 0);
    lv_obj_align(warning_title, LV_ALIGN_TOP_MID, 0, 0);

    veh_warning_panel = lv_obj_create(warning_panel_container);
    lv_obj_set_size(veh_warning_panel, lv_pct(95), 45);
    lv_obj_set_style_bg_color(veh_warning_panel, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(veh_warning_panel, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(veh_warning_panel, 1, 0);
    lv_obj_set_style_radius(veh_warning_panel, CORNER_RADIUS / 2, 0);
    lv_obj_align_to(veh_warning_panel, warning_title, LV_ALIGN_OUT_BOTTOM_MID, 0, 0);

    auto mk_light = [&](int xOff, const char *txt, uint32_t col, const lv_font_t *font)
    {
        lv_obj_t *light = lv_obj_create(veh_warning_panel);
        lv_obj_set_size(light, 35, 35);
        lv_obj_set_style_bg_color(light, color_from_hex(EPAPER_WHITE), 0);
        lv_obj_set_style_border_color(light, color_from_hex(EPAPER_BLACK), 0);
        lv_obj_set_style_border_width(light, 2, 0);
        lv_obj_set_style_radius(light, 17, 0);
        lv_obj_align(light, LV_ALIGN_LEFT_MID, xOff, 0);

        lv_obj_t *lab = lv_label_create(light);
        lv_label_set_text(lab, txt);
        lv_obj_set_style_text_color(lab, color_from_hex(col), 0);
        lv_obj_set_style_text_font(lab, font, 0);
        lv_obj_center(lab);
    };

    mk_light(20, "ENG", EPAPER_BLACK, &lv_font_montserrat_12);
    mk_light(80, "OIL", EPAPER_BLACK, &lv_font_montserrat_12);
    mk_light(140, "BAT", EPAPER_BLACK, &lv_font_montserrat_12);
    mk_light(200, "BRK", EPAPER_BLACK, &lv_font_montserrat_12);
    mk_light(260, "◄", EPAPER_YELLOW, &lv_font_montserrat_16);
    mk_light(320, "►", EPAPER_YELLOW, &lv_font_montserrat_16);

    lv_obj_t *highbeam = lv_obj_create(veh_warning_panel);
    lv_obj_set_size(highbeam, 35, 35);
    lv_obj_set_style_bg_color(highbeam, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_border_color(highbeam, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(highbeam, 2, 0);
    lv_obj_set_style_radius(highbeam, 17, 0);
    lv_obj_align(highbeam, LV_ALIGN_LEFT_MID, 380, 0);

    lv_obj_t *hi_label = lv_label_create(highbeam);
    lv_label_set_text(hi_label, "HI");
    lv_obj_set_style_text_color(hi_label, color_from_hex(EPAPER_BLUE), 0);
    lv_obj_set_style_text_font(hi_label, &lv_font_montserrat_12, 0);
    lv_obj_center(hi_label);

    lv_obj_t *seatbelt = lv_obj_create(veh_warning_panel);
    lv_obj_set_size(seatbelt, 35, 35);
    lv_obj_set_style_bg_color(seatbelt, color_from_hex(EPAPER_RED), 0);
    lv_obj_set_style_border_color(seatbelt, color_from_hex(EPAPER_BLACK), 0);
    lv_obj_set_style_border_width(seatbelt, 2, 0);
    lv_obj_set_style_radius(seatbelt, 17, 0);
    lv_obj_align(seatbelt, LV_ALIGN_LEFT_MID, 440, 0);

    lv_obj_t *belt_label = lv_label_create(seatbelt);
    lv_label_set_text(belt_label, "BELT");
    lv_obj_set_style_text_color(belt_label, color_from_hex(EPAPER_WHITE), 0);
    lv_obj_set_style_text_font(belt_label, &lv_font_montserrat_12, 0);
    lv_obj_center(belt_label);

    // Store containers for updates
    veh_speed_gauge = speed_gauge_container;
    veh_rpm_gauge = rpm_gauge_container;
}

/* Vehicle updates */
void update_speed_gauge(int speed)
{
    if (speed < 0)
        speed = 0;
    if (speed > 200)
        speed = 200;

    lv_label_set_text_fmt(veh_speed_label, "%d", speed);

    float angle = -135.0f + (270.0f * speed / 200.0f);
    float rad = angle * 3.14159f / 180.0f;

    lv_obj_t *speed_bg = lv_obj_get_child(veh_speed_gauge, 1);
    lv_obj_t *needle = lv_obj_get_child(speed_bg, lv_obj_get_child_cnt(speed_bg) - 2);

    static lv_point_precise_t needle_points[2];
    int large_radius = LARGE_GAUGE_SIZE / 2 - 15;
    needle_points[0].x = LARGE_GAUGE_SIZE / 2;
    needle_points[0].y = LARGE_GAUGE_SIZE / 2;
    needle_points[1].x = LARGE_GAUGE_SIZE / 2 + (large_radius - 10) * cos(rad);
    needle_points[1].y = LARGE_GAUGE_SIZE / 2 + (large_radius - 10) * sin(rad);

    lv_line_set_points(needle, needle_points, 2);
}

void update_rpm_gauge(int rpm)
{
    if (rpm < 0)
        rpm = 0;
    if (rpm > 8000)
        rpm = 8000;

    lv_label_set_text_fmt(veh_rpm_label, "%d", rpm);

    float angle = -135.0f + (270.0f * rpm / 8000.0f);
    float rad = angle * 3.14159f / 180.0f;

    lv_obj_t *rpm_bg = lv_obj_get_child(veh_rpm_gauge, 1);
    lv_obj_t *needle = lv_obj_get_child(rpm_bg, lv_obj_get_child_cnt(rpm_bg) - 2);

    static lv_point_precise_t needle_points[2];
    int large_radius = LARGE_GAUGE_SIZE / 2 - 15;
    needle_points[0].x = LARGE_GAUGE_SIZE / 2;
    needle_points[0].y = LARGE_GAUGE_SIZE / 2;
    needle_points[1].x = LARGE_GAUGE_SIZE / 2 + (large_radius - 10) * cos(rad);
    needle_points[1].y = LARGE_GAUGE_SIZE / 2 + (large_radius - 10) * sin(rad);

    lv_line_set_points(needle, needle_points, 2);
}

void update_fuel_level(int percentage)
{
    if (percentage < 0)
        percentage = 0;
    if (percentage > 100)
        percentage = 100;
    static char buf[32];
    lv_snprintf(buf, sizeof(buf), "%d%%", percentage);
    lv_label_set_text(veh_fuel_label, buf);
    lv_color_t color = (percentage <= 15) ? color_from_hex(EPAPER_RED) : color_from_hex(EPAPER_BLACK);
    lv_obj_set_style_text_color(veh_fuel_label, color, 0);
}

void update_engine_temp(int temperature)
{
    static char buf[32];
    lv_snprintf(buf, sizeof(buf), "%d°C", temperature);
    lv_label_set_text(veh_temp_label, buf);
    lv_color_t color = (temperature >= 100) ? color_from_hex(EPAPER_RED) : color_from_hex(EPAPER_BLACK);
    lv_obj_set_style_text_color(veh_temp_label, color, 0);
}

void update_gear_position(char gear)
{
    char gear_str[2] = {gear, '\0'};
    lv_label_set_text(veh_gear_label, gear_str);
}

void update_odometer(long distance)
{
    static char buf[32];
    lv_snprintf(buf, sizeof(buf), "%6ld km", distance);
    lv_label_set_text(veh_odometer_label, buf);
}

void set_warning_indicator(uint8_t indicator_id, bool state)
{
    lv_obj_t *indicator = lv_obj_get_child(veh_warning_panel, indicator_id);
    if (indicator != NULL)
    {
        lv_obj_set_style_bg_color(indicator, state ? color_from_hex(EPAPER_RED) : color_from_hex(EPAPER_WHITE), 0);
    }
}
