#!/usr/bin/env bash
# Auto flash latest .bin (or .hex) in current directory using xiao_samd11_openocd_flash.py
# Usage:
#   ./flash.sh              -> auto-pick newest .bin (fallback newest .hex)
#   ./flash.sh myimage.bin  -> use specified file
# Env overrides:
#   PY=python3            (python interpreter; default: python)
#   FLASH_SPEED=1000      (adapter speed kHz)
#   FLASH_BASE=0x00000000 (base address for .bin)
#   NO_AUTO_INSTALL=1     (disable --auto-install-openocd)
#   NO_RESET=1            (omit --reset-after)
#   SERIAL_PORT=/dev/ttyACM0 (open serial after flash)
#   NO_SERIAL_AUTO=1      (disable auto single-port pick)
#
# Exit codes mirror python script where possible plus:
#   10: no image found
#
set -euo pipefail
SCRIPT="xiao_samd11_openocd_flash.py"
PY=${PY:-python}

if [[ ! -f "$SCRIPT" ]]; then
  echo "[ERROR] Script $SCRIPT not found in $(pwd)" >&2
  exit 1
fi

IMAGE="${1:-}"
if [[ -z "$IMAGE" ]]; then
  # newest .bin
  IMAGE=$(ls -1t *.bin 2>/dev/null | head -n1 || true)
  if [[ -z "$IMAGE" ]]; then
    IMAGE=$(ls -1t *.hex 2>/dev/null | head -n1 || true)
  fi
fi
if [[ -z "$IMAGE" ]]; then
  echo "[ERROR] No .bin or .hex file found." >&2
  exit 10
fi

BASE=${FLASH_BASE:-0x00000000}
SPEED=${FLASH_SPEED:-1000}
RESET_FLAG="--reset-after"
if [[ "${NO_RESET:-}" == 1 ]]; then
  RESET_FLAG=""
fi
AUTO_INSTALL_FLAG="--auto-install-openocd"
if [[ "${NO_AUTO_INSTALL:-}" == 1 ]]; then
  AUTO_INSTALL_FLAG=""
fi
SERIAL_ARGS=()
if [[ -n "${SERIAL_PORT:-}" ]]; then
  SERIAL_ARGS+=(--serial-port "${SERIAL_PORT}")
fi
if [[ "${NO_SERIAL_AUTO:-}" == 1 ]]; then
  SERIAL_ARGS+=(--no-serial-auto)
fi

BASE_ARG=()
ext=${IMAGE##*.}
shopt -s nocasematch || true
if [[ "$ext" == "bin" ]]; then
  BASE_ARG=(--base-addr "$BASE")
fi
shopt -u nocasematch || true

echo "[INFO] Using image: $IMAGE"
echo "[INFO] Adapter speed: $SPEED kHz"
echo "[INFO] Invoking: $PY $SCRIPT --file $IMAGE --openocd-speed $SPEED $RESET_FLAG $AUTO_INSTALL_FLAG ${BASE_ARG[*]} ${SERIAL_ARGS[*]}"

set +e
$PY "$SCRIPT" --file "$IMAGE" --openocd-speed "$SPEED" ${RESET_FLAG} ${AUTO_INSTALL_FLAG} ${BASE_ARG[@]} "${SERIAL_ARGS[@]}"
rc=$?
set -e
if [[ $rc -ne 0 ]]; then
  echo "[ERROR] Flash operation failed. (exit code $rc)" >&2
  exit $rc
fi

echo "[OK] Flash operation completed successfully."
exit 0
