#include <TFT_eSPI.h>
#include <lvgl.h>
#include <math.h>

#include "Adafruit_SHT4x.h"

#include "e1002_display.h"
#include "ui/GUI.h"

#include <WiFi.h>

#include "weather.h"
#include "weather_view.h"

e1002_driver_t e1002_driver;
Adafruit_SHT4x sht4 = Adafruit_SHT4x();
// ==================== 配置 ====================
const char *WIFI_SSID = "*************";
const char *WIFI_PASSWORD = "*************";
const char *OW_API_KEY = "**************************";    // OpenWeather API Key

const char *UNITS = "metric";
const char *LANG = "zh_cn";

const uint32_t WEATHER_REFRESH_INTERVAL_MIN = 60;  // Weather API call interval
const uint32_t INDOOR_REFRESH_INTERVAL_MIN = 10;   // Indoor sensor/UI refresh interval
const uint32_t WEATHER_REFRESH_INTERVAL_MS = WEATHER_REFRESH_INTERVAL_MIN * 60UL * 1000UL;
const uint32_t INDOOR_REFRESH_INTERVAL_MS = INDOOR_REFRESH_INTERVAL_MIN * 60UL * 1000UL;

// ✅ 方式1：城市名
String CITY = "Shenzhen,CN"; // 推荐带国家码避免同名冲突

// ✅ 方式2：经纬度
const char *LAT = "40.7128";
const char *LON = "-74.0060";

// 开关：true = 使用城市名；false = 使用经纬度
bool USE_CITY_MODE = true;

typedef struct {
    float    temperature;
    float    humidity;
} SHT40_DATA;

WeatherRequest g_weatherRequest;
WeatherData g_weatherData;
bool g_weatherValid = false;
uint32_t g_lastWeatherFetchMs = 0;
uint32_t g_lastIndoorUpdateMs = 0;

SHT40_DATA sht40_read()
{
    sensors_event_t humidity, temp;
    sht4.getEvent(&humidity, &temp);
    SHT40_DATA d;
    d.temperature   = temp.temperature;
    d.humidity      = humidity.relative_humidity;
    return d;
}

// ==================== 主函数 ====================
void setup()
{
    Serial1.begin(115200, SERIAL_8N1, 44, 43);
    e1002_display_init(&e1002_driver);
    GUI_init();
    Wire.setPins(19, 20);
    sht4.begin();

    WiFi.begin(WIFI_SSID, WIFI_PASSWORD);
    Serial1.print("Connecting to WiFi...");
    while (WiFi.status() != WL_CONNECTED)
    {
        delay(500);
        Serial1.print(".");
    }
    Serial1.println(" Connected!");
    Serial1.println("IP address: " + WiFi.localIP().toString());

    g_weatherRequest.apiKey = OW_API_KEY;
    g_weatherRequest.units = UNITS;
    g_weatherRequest.lang = LANG;
    g_weatherRequest.useCity = USE_CITY_MODE;
    if (USE_CITY_MODE)
    {
        g_weatherRequest.city = CITY;
    }
    else
    {
        g_weatherRequest.lat = LAT;
        g_weatherRequest.lon = LON;
    }

    if (fetchWeather(g_weatherRequest, g_weatherData))
    {
        Serial1.println("=== CITY/TZ ===");
        Serial1.printf("city=%s tzOffset=%d(s)\n", g_weatherData.cityName.c_str(), g_weatherData.timezoneOffset);

        Serial1.println("=== CURRENT ===");
        Serial1.printf("temp=%.1f feels=%.1f hum=%d%% wind=%.1f gust=%.1f deg=%d\n",
                       g_weatherData.temp, g_weatherData.feelsLike, g_weatherData.humidity,
                       g_weatherData.windSpeed, g_weatherData.windGust, g_weatherData.windDeg);
        Serial1.printf("weather: %d %s / %s icon=%s\n",
                       g_weatherData.weatherId, g_weatherData.main.c_str(), g_weatherData.desc.c_str(), g_weatherData.icon.c_str());

        Serial1.println("=== HOURLY (next 4 x 3h) ===");
        for (int i = 0; i < 4; i++)
        {
            Serial1.printf("[%d] t=%.1f wind=%.1f gust=%.1f id=%d %s icon=%s\n",
                           g_weatherData.hourly[i].ts, g_weatherData.hourly[i].temp, g_weatherData.hourly[i].windSpeed,
                           g_weatherData.hourly[i].windGust, g_weatherData.hourly[i].weatherId,
                           g_weatherData.hourly[i].main.c_str(), g_weatherData.hourly[i].icon.c_str());
        }

        Serial1.println("=== DAILY (next 3, aggregated) ===");
        for (int i = 0; i < 3; i++)
        {
            Serial1.printf("[%d] day=%.1f min=%.1f max=%.1f id=%d %s icon=%s\n",
                           g_weatherData.daily[i].ts, g_weatherData.daily[i].day, g_weatherData.daily[i].min, g_weatherData.daily[i].max,
                           g_weatherData.daily[i].weatherId, g_weatherData.daily[i].main.c_str(), g_weatherData.daily[i].icon.c_str());
        }

        Serial1.println("=== AQI ===");
        Serial1.printf("AQI=%d (%s)\n", g_weatherData.aqi, g_weatherData.aqiLabel.c_str());

        Serial1.println("Alerts: not available on 2.5");

        SHT40_DATA indoor = sht40_read();
        g_weatherData.indoorTemp = indoor.temperature;
        g_weatherData.indoorHumidity = static_cast<int>(roundf(indoor.humidity));

        GUI_update_weather_screen(g_weatherData);
        g_weatherValid = true;
        g_lastWeatherFetchMs = millis();
        g_lastIndoorUpdateMs = g_lastWeatherFetchMs;
    }
    else
    {
        Serial1.println("Fetch weather failed.");
    }
}

void loop()
{
    lv_timer_handler(); // Drive LVGL internal timers
    delay(50);          // Reduce CPU usage since frequent updates aren't needed

    const uint32_t now = millis();

    if (!g_weatherValid || (now - g_lastWeatherFetchMs) >= WEATHER_REFRESH_INTERVAL_MS)
    {
        WeatherData latest;
        if (fetchWeather(g_weatherRequest, latest))
        {
            // Preserve latest indoor readings
            latest.indoorTemp = g_weatherData.indoorTemp;
            latest.indoorHumidity = g_weatherData.indoorHumidity;

            g_weatherData = latest;
            g_weatherValid = true;
            g_lastWeatherFetchMs = now;
            GUI_update_weather_screen(g_weatherData);
        }
    }

    if ((now - g_lastIndoorUpdateMs) >= INDOOR_REFRESH_INTERVAL_MS)
    {
        SHT40_DATA indoor = sht40_read();
        if (g_weatherValid)
        {
            g_weatherData.indoorTemp = indoor.temperature;
            g_weatherData.indoorHumidity = static_cast<int>(roundf(indoor.humidity));
            GUI_update_weather_screen(g_weatherData);
        }
        g_lastIndoorUpdateMs = now;
    }

    // Check if display refresh is needed
    if (e1002_display_should_refresh(&e1002_driver))
    {
        Serial1.println("Refreshing e-paper display...");
        e1002_display_refresh(&e1002_driver);
        Serial1.println("Display refresh complete");
    }
}
