#include "weather.h"

#include <WiFi.h>
#include <HTTPClient.h>
#include <ArduinoJson.h>
#include <map>
#include <vector>
#include <algorithm>
#include <limits.h>
#include <time.h>

namespace
{
String aqiLabelFrom(int aqi)
{
    switch (aqi)
    {
    case 1: return "Good";
    case 2: return "Fair";
    case 3: return "Moderate";
    case 4: return "Poor";
    case 5: return "Very Poor";
    default: return "Unknown";
    }
}

String urlEncode(const String &s)
{
    String o;
    o.reserve(s.length() * 3);
    for (size_t i = 0; i < s.length(); ++i)
    {
        char c = s[i];
        if (c == ' ')
            o += "%20";
        else
            o += c;
    }
    return o;
}

String weatherByCityUrl(const WeatherRequest &req)
{
    return String("https://api.openweathermap.org/data/2.5/weather") +
           "?q=" + urlEncode(req.city) +
           "&units=" + req.units +
           "&lang=" + req.lang +
           "&appid=" + req.apiKey;
}

String weatherByCoordUrl(const WeatherRequest &req, const String &lat, const String &lon)
{
    return String("https://api.openweathermap.org/data/2.5/weather") +
           "?lat=" + lat + "&lon=" + lon +
           "&units=" + req.units + "&lang=" + req.lang +
           "&appid=" + req.apiKey;
}

String forecastByCoordUrl(const WeatherRequest &req, float lat, float lon)
{
    return String("https://api.openweathermap.org/data/2.5/forecast") +
           "?lat=" + String(lat, 6) + "&lon=" + String(lon, 6) +
           "&units=" + req.units + "&lang=" + req.lang +
           "&appid=" + req.apiKey;
}

String airByCoordUrl(const WeatherRequest &req, float lat, float lon)
{
    return String("https://api.openweathermap.org/data/2.5/air_pollution") +
           "?lat=" + String(lat, 6) + "&lon=" + String(lon, 6) +
           "&appid=" + req.apiKey;
}

bool httpGetJson(const String &url, DynamicJsonDocument &doc)
{
    HTTPClient http;
    WiFiClientSecure client;
    client.setInsecure();
    http.begin(client, url);
    int code = http.GET();
    String body = http.getString();
    if (code != HTTP_CODE_OK)
    {
        Serial1.printf("HTTP GET %s -> %d\n", url.c_str(), code);
        Serial1.println(body);
        http.end();
        return false;
    }
    DeserializationError err = deserializeJson(doc, body);
    http.end();
    if (err)
    {
        Serial1.printf("JSON parse error: %s\n", err.c_str());
        return false;
    }
    return true;
}

struct Ymd
{
    int y;
    int m;
    int d;
};

Ymd ymdFromTs(int64_t tsUtc, int tzOffset)
{
    time_t t = tsUtc + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    return {tmv.tm_year + 1900, tmv.tm_mon + 1, tmv.tm_mday};
}

int secondsOfDay(int64_t tsUtc, int tzOffset)
{
    time_t t = tsUtc + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    return tmv.tm_hour * 3600 + tmv.tm_min * 60 + tmv.tm_sec;
}

void aggregateNext3Days(const DynamicJsonDocument &fcDoc, WeatherData &out)
{
    out.cityName = (const char *)(fcDoc["city"]["name"] | "");
    out.timezoneOffset = fcDoc["city"]["timezone"] | 0;
    JsonArrayConst list = fcDoc["list"].as<JsonArrayConst>();
    if (list.isNull() || list.size() == 0)
        return;

    struct DayAgg
    {
        bool inited = false;
        uint32_t anyTs = 0;
        float minT = 1e9;
        float maxT = -1e9;
        int bestDelta = INT_MAX;
        float dayT = NAN;
        int wid = 0;
        String main;
        String desc;
        String icon;
    };

    std::map<String, DayAgg> agg;
    for (JsonVariantConst itVar : list)
    {
        JsonObjectConst it = itVar.as<JsonObjectConst>();
        int64_t ts = it["dt"] | 0;
        float tt = it["main"]["temp"] | NAN;
        int wid = it["weather"][0]["id"] | 0;
        String main = (const char *)(it["weather"][0]["main"] | "");
        String desc = (const char *)(it["weather"][0]["description"] | "");
        String icon = (const char *)(it["weather"][0]["icon"] | "");

        Ymd ymd = ymdFromTs(ts, out.timezoneOffset);
        char keyBuf[16];
        snprintf(keyBuf, sizeof(keyBuf), "%04d-%02d-%02d", ymd.y, ymd.m, ymd.d);
        String key(keyBuf);
        DayAgg &d = agg[key];
        if (!d.inited)
        {
            d.inited = true;
            d.anyTs = ts;
        }
        if (!isnan(tt))
        {
            if (tt < d.minT)
                d.minT = tt;
            if (tt > d.maxT)
                d.maxT = tt;
        }
        int delta = abs(secondsOfDay(ts, out.timezoneOffset) - 12 * 3600);
        if (delta < d.bestDelta)
        {
            d.bestDelta = delta;
            d.dayT = tt;
            d.wid = wid;
            d.main = main;
            d.desc = desc;
            d.icon = icon;
        }
    }

    std::vector<std::pair<String, DayAgg>> days;
    for (auto &kv : agg)
        days.push_back(kv);
    std::sort(days.begin(), days.end(),
              [](const std::pair<String, DayAgg> &a, const std::pair<String, DayAgg> &b) {
                  return a.first.compareTo(b.first) < 0;
              });

    int idx = 0;
    for (auto &kv : days)
    {
        if (idx >= 3)
            break;
        DayAgg &d = kv.second;
        out.daily[idx].ts = d.anyTs;
        out.daily[idx].day = d.dayT;
        out.daily[idx].min = d.minT;
        out.daily[idx].max = d.maxT;
        out.daily[idx].weatherId = d.wid;
        out.daily[idx].main = d.main;
        out.daily[idx].desc = d.desc;
        out.daily[idx].icon = d.icon;
        idx++;
    }
}

} // namespace

bool fetchWeather(const WeatherRequest &request, WeatherData &out)
{
    if (request.apiKey.isEmpty())
    {
        Serial1.println("Weather API key is empty.");
        return false;
    }

    WeatherData data;
    float lat = 0;
    float lon = 0;

    if (request.useCity)
    {
        if (request.city.isEmpty())
        {
            Serial1.println("City name not provided.");
            return false;
        }

        DynamicJsonDocument doc(24 * 1024);
        if (!httpGetJson(weatherByCityUrl(request), doc))
            return false;

        data.cityName = (const char *)(doc["name"] | request.city.c_str());
        lat = doc["coord"]["lat"] | 0.0;
        lon = doc["coord"]["lon"] | 0.0;

        data.dt = doc["dt"] | 0;
        data.temp = doc["main"]["temp"] | NAN;
        data.feelsLike = doc["main"]["feels_like"] | NAN;
        data.humidity = doc["main"]["humidity"] | 0;
        data.windSpeed = doc["wind"]["speed"] | 0.0;
        data.windDeg = doc["wind"]["deg"] | 0;
        data.windGust = doc["wind"]["gust"] | 0.0;
        JsonObject w0 = doc["weather"][0].as<JsonObject>();
        data.weatherId = w0["id"] | 0;
        data.main = (const char *)(w0["main"] | "");
        data.desc = (const char *)(w0["description"] | "");
        data.icon = (const char *)(w0["icon"] | "");
    }
    else
    {
        if (request.lat.isEmpty() || request.lon.isEmpty())
        {
            Serial1.println("Latitude/Longitude not provided.");
            return false;
        }

        lat = request.lat.toFloat();
        lon = request.lon.toFloat();

        DynamicJsonDocument doc(16 * 1024);
        if (!httpGetJson(weatherByCoordUrl(request, request.lat, request.lon), doc))
            return false;

        data.cityName = (const char *)(doc["name"] | "");
        data.dt = doc["dt"] | 0;
        data.temp = doc["main"]["temp"] | NAN;
        data.feelsLike = doc["main"]["feels_like"] | NAN;
        data.humidity = doc["main"]["humidity"] | 0;
        data.windSpeed = doc["wind"]["speed"] | 0.0;
        data.windDeg = doc["wind"]["deg"] | 0;
        data.windGust = doc["wind"]["gust"] | 0.0;
        JsonObject w0 = doc["weather"][0].as<JsonObject>();
        data.weatherId = w0["id"] | 0;
        data.main = (const char *)(w0["main"] | "");
        data.desc = (const char *)(w0["description"] | "");
        data.icon = (const char *)(w0["icon"] | "");
    }

    {
        DynamicJsonDocument doc(96 * 1024);
        if (!httpGetJson(forecastByCoordUrl(request, lat, lon), doc))
            return false;

        data.cityName = (const char *)(doc["city"]["name"] | data.cityName.c_str());
        data.timezoneOffset = doc["city"]["timezone"] | 0;

        JsonArray list = doc["list"].as<JsonArray>();
        for (int i = 0; i < 4 && i < (int)list.size(); ++i)
        {
            JsonObject it = list[i].as<JsonObject>();
            data.hourly[i].ts = it["dt"] | 0;
            data.hourly[i].temp = it["main"]["temp"] | NAN;
            data.hourly[i].windSpeed = it["wind"]["speed"] | 0.0;
            data.hourly[i].windGust = it["wind"]["gust"] | 0.0;
            JsonObject ww = it["weather"][0];
            data.hourly[i].weatherId = ww["id"] | 0;
            data.hourly[i].main = (const char *)(ww["main"] | "");
            data.hourly[i].desc = (const char *)(ww["description"] | "");
            data.hourly[i].icon = (const char *)(ww["icon"] | "");
        }

        aggregateNext3Days(doc, data);
    }

    {
        DynamicJsonDocument doc(8 * 1024);
        if (httpGetJson(airByCoordUrl(request, lat, lon), doc))
        {
            int aqi = doc["list"][0]["main"]["aqi"] | 0;
            data.aqi = aqi;
            data.aqiLabel = aqiLabelFrom(aqi);
        }
        else
        {
            data.aqi = 0;
            data.aqiLabel = "Unknown";
        }
    }

    data.hasAlert = false;
    out = data;
    return true;
}
