#include "weather_view.h"

#include "ui/GUI.h"

#include <lvgl.h>
#include <algorithm>
#include <math.h>
#include <time.h>

namespace
{
String formatTemperature(float value)
{
    if (isnan(value))
        return "--";
    int rounded = static_cast<int>(roundf(value));
    return String(rounded) + "°";
}

String formatHumidity(int value)
{
    if (value < 0)
        return "--";
    return String(value) + "%";
}

String formatWindKmH(float speedMs)
{
    if (isnan(speedMs))
        return "--";
    float kmh = speedMs * 3.6f;
    int rounded = static_cast<int>(roundf(kmh));
    return String(rounded);
}

String formatHour(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    char buf[6];
    snprintf(buf, sizeof(buf), "%02d:%02d", tmv.tm_hour, tmv.tm_min);
    return String(buf);
}

String formatWeekdayShort(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    static const char *kWeekdays[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    int idx = tmv.tm_wday;
    if (idx < 0 || idx > 6)
        return "--";
    return String(kWeekdays[idx]);
}

String formatWeekdayLong(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    static const char *kWeekdays[] = {"Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"};
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    int idx = tmv.tm_wday;
    if (idx < 0 || idx > 6)
        return "--";
    return String(kWeekdays[idx]);
}

String formatDateLine(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    char buf[16];
    // Match original layout: DD.MM.YYYY
    snprintf(buf, sizeof(buf), "%02d.%02d.%04d", tmv.tm_mday, tmv.tm_mon + 1, tmv.tm_year + 1900);
    return String(buf);
}

String formatDailyTemp(const DayItem &day)
{
    if (isnan(day.min) && isnan(day.max) && isnan(day.day))
        return "--";

    float base = !isnan(day.day) ? day.day : day.max;
    if (isnan(base))
        base = day.min;
    if (isnan(base))
        return "--";
    return formatTemperature(base);
}

String formatTempRange(float minVal, float maxVal)
{
    if (isnan(minVal) || isnan(maxVal))
        return "--";
    if (maxVal < minVal)
        std::swap(minVal, maxVal);
    int minRounded = static_cast<int>(roundf(minVal));
    int maxRounded = static_cast<int>(roundf(maxVal));
    return String(minRounded) + "° - " + String(maxRounded) + "°";
}

String fallbackText(const String &value, const char *fallback = "--")
{
    return value.length() > 0 ? value : String(fallback);
}

int aqiPercentFromLevel(int aqi)
{
    switch (aqi)
    {
    case 1: return 100;
    case 2: return 80;
    case 3: return 60;
    case 4: return 40;
    case 5: return 20;
    default: return 0;
    }
}

int percentFromRange(float value, float minValue, float maxValue)
{
    if (isnan(value) || maxValue <= minValue)
        return 0;
    float norm = (value - minValue) / (maxValue - minValue);
    int percent = static_cast<int>(roundf(norm * 100.0f));
    if (percent < 0)
        percent = 0;
    if (percent > 100)
        percent = 100;
    return percent;
}

int clampPercent(int value)
{
    if (value < 0)
        return 0;
    if (value > 100)
        return 100;
    return value;
}

} // namespace

void GUI_update_weather_screen(const WeatherData &data)
{
    // Current status and headline
    lv_label_set_text(GUI_Label__main_screen__forecast_current_status_lbl,
                      fallbackText(data.main, "Loading...").c_str());

    lv_label_set_text(GUI_Label__main_screen__current_day_city_lbl,
                      fallbackText(data.cityName, "--").c_str());
    lv_label_set_text(GUI_Label__main_screen__current_day_date_lbl,
                      formatDateLine(data.dt, data.timezoneOffset).c_str());
    lv_label_set_text(GUI_Label__main_screen__current_day_weekday_lbl,
                      formatWeekdayLong(data.dt, data.timezoneOffset).c_str());
    lv_label_set_text(GUI_Label__main_screen__current_day_temp_lbl,
                      formatTemperature(data.temp).c_str());
    lv_label_set_text(GUI_Label__main_screen__current_day_status_lbl,
                      fallbackText(data.main).c_str());

    // Hourly forecast
    lv_obj_t *const hourlyTimeLabels[4] = {
        GUI_Label__main_screen__forecast_hourly_item_1_time_lbl,
        GUI_Label__main_screen__forecast_hourly_item_2_time_lbl,
        GUI_Label__main_screen__forecast_hourly_item_3_time_lbl,
        GUI_Label__main_screen__forecast_hourly_item_4_time_lbl,
    };
    lv_obj_t *const hourlyTempLabels[4] = {
        GUI_Label__main_screen__forecast_hourly_item_1_temp_lbl,
        GUI_Label__main_screen__forecast_hourly_item_2_temp_lbl,
        GUI_Label__main_screen__forecast_hourly_item_3_temp_lbl,
        GUI_Label__main_screen__forecast_hourly_item_4_temp_lbl,
    };

    for (int i = 0; i < 4; ++i)
    {
        const HourItem &item = data.hourly[i];
        lv_label_set_text(hourlyTimeLabels[i], formatHour(item.ts, data.timezoneOffset).c_str());
        lv_label_set_text(hourlyTempLabels[i], formatTemperature(item.temp).c_str());
    }

    // Daily forecast
    lv_obj_t *const dailyDayLabels[3] = {
        GUI_Label__main_screen__forecast_daily_item_1_day_lbl,
        GUI_Label__main_screen__forecast_daily_item_2_day_lbl,
        GUI_Label__main_screen__forecast_daily_item_3_day_lbl,
    };
    lv_obj_t *const dailyTempLabels[3] = {
        GUI_Label__main_screen__forecast_daily_item_1_temp_lbl,
        GUI_Label__main_screen__forecast_daily_item_2_temp_lbl,
        GUI_Label__main_screen__forecast_daily_item_3_temp_lbl,
    };

    for (int i = 0; i < 3; ++i)
    {
        const DayItem &item = data.daily[i];
        lv_label_set_text(dailyDayLabels[i], formatWeekdayShort(item.ts, data.timezoneOffset).c_str());
        lv_label_set_text(dailyTempLabels[i], formatDailyTemp(item).c_str());
    }

    // Outdoor summary
    lv_label_set_text(GUI_Label__main_screen__details_outdoor_humidity_value_lbl,
                      formatHumidity(data.humidity).c_str());
    // String humidityRange = "--";
    // if (!isnan(data.daily[0].min) && !isnan(data.daily[0].max))
        // humidityRange = formatTempRange(data.daily[0].min, data.daily[0].max);
    // lv_label_set_text(GUI_Label__main_screen__details_outdoor_humidity_range_lbl, humidityRange.c_str());

    lv_label_set_text(GUI_Label__main_screen__details_outdoor_wind_value_lbl,
                      formatWindKmH(data.windSpeed).c_str());
    lv_label_set_text(GUI_Label__main_screen__details_outdoor_wind_unit_lbl, "km/h");

    // Indoor stats
    lv_label_set_text(GUI_Label__main_screen__details_indoor_temp_value_lbl,
                      formatTemperature(data.indoorTemp).c_str());

    const int aqiPercent = aqiPercentFromLevel(data.aqi);
    const int indoorTempPercent = percentFromRange(data.indoorTemp, 0.0f, 35.0f);
    const int indoorHumidityPercent = data.indoorHumidity >= 0 ? clampPercent(data.indoorHumidity) : 0;

    lv_bar_set_value(GUI_Bar__main_screen__details_indoor_temp_indicator_bar, indoorTempPercent, LV_ANIM_OFF);
    lv_bar_set_value(GUI_Bar__main_screen__details_indoor_humidity_indicator_bar, indoorHumidityPercent, LV_ANIM_OFF);
    lv_bar_set_value(GUI_Bar__main_screen__details_indoor_fan_indicator_bar, aqiPercent, LV_ANIM_OFF);

    lv_label_set_text(GUI_Label__main_screen__details_indoor_air_quality_title_lbl, "Air Quality");
    String aqiText = fallbackText(data.aqiLabel, "");
    if (aqiPercent > 0)
    {
        if (aqiText.length() > 0)
            aqiText += " · ";
        aqiText += String(aqiPercent) + "%";
    }
    if (aqiText.length() == 0)
        aqiText = "--";
    lv_label_set_text(GUI_Label__main_screen__details_indoor_air_quality_value_lbl, aqiText.c_str());

    // Alert message
    if (data.hasAlert)
    {
        lv_label_set_text(GUI_Label__main_screen__alert_title_lbl,
                          fallbackText(data.alert.event, "Weather alert").c_str());
        lv_label_set_text(GUI_Label__main_screen__alert_details_lbl,
                          fallbackText(data.alert.description, "Details unavailable").c_str());
    }
    else
    {
        lv_label_set_text(GUI_Label__main_screen__alert_title_lbl, "No alerts");
        lv_label_set_text(GUI_Label__main_screen__alert_details_lbl, "No severe alerts.");
    }
}
