#include "weather_view.h"

#include "ui/GUI.h"

#include <lvgl.h>
#include <math.h>
#include <time.h>

namespace
{
String formatTemperature(float value)
{
    if (isnan(value))
        return "--";
    int rounded = static_cast<int>(roundf(value));
    return String(rounded) + "°";
}

String formatHumidity(int value)
{
    if (value < 0)
        return "--";
    return String(value) + "%";
}

String formatWind(float speedMs)
{
    if (isnan(speedMs))
        return "--";
    float kmh = speedMs * 3.6f;
    int rounded = static_cast<int>(roundf(kmh));
    return String(rounded);
}

String formatHour(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    char buf[6];
    snprintf(buf, sizeof(buf), "%02d:%02d", tmv.tm_hour, tmv.tm_min);
    return String(buf);
}

String formatWeekday(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    static const char *kWeekdays[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    int idx = tmv.tm_wday;
    if (idx < 0 || idx > 6)
        return "--";
    return String(kWeekdays[idx]);
}

String formatDateLine(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    char buf[16];
    // Match original layout: DD.MM.YYYY
    snprintf(buf, sizeof(buf), "%02d.%02d.%04d", tmv.tm_mday, tmv.tm_mon + 1, tmv.tm_year + 1900);
    return String(buf);
}

String formatDailyTemp(const DayItem &day)
{
    if (isnan(day.min) && isnan(day.max) && isnan(day.day))
        return "--";

    if (!isnan(day.min) && !isnan(day.max))
    {
        int minVal = static_cast<int>(roundf(day.min));
        int maxVal = static_cast<int>(roundf(day.max));
        // return String(maxVal) + "°/" + String(minVal) + "°";
        return String(maxVal) + "°";
    }

    if (!isnan(day.day))
    {
        int dayVal = static_cast<int>(roundf(day.day));
        return String(dayVal) + "°";
    }

    return "--";
}

String fallbackText(const String &value, const char *fallback = "--")
{
    return value.length() > 0 ? value : String(fallback);
}

int aqiPercentFromLevel(int aqi)
{
    switch (aqi)
    {
    case 1: return 100;
    case 2: return 80;
    case 3: return 60;
    case 4: return 40;
    case 5: return 20;
    default: return 0;
    }
}

} // namespace

void GUI_update_weather_screen(const WeatherData &data)
{
    // Current status
    lv_label_set_text(GUI_Label__cont__current_status_text_lbl,
                      fallbackText(data.main, "Loading...").c_str());

    // Main temperature and location/date
    lv_label_set_text(GUI_Label__cont__main_display_temp_lbl, formatTemperature(data.temp).c_str());

    String titleLine = formatDateLine(data.dt, data.timezoneOffset);
    titleLine += "\n";
    titleLine += fallbackText(data.cityName, "Unknown");
    lv_label_set_text(GUI_Label__cont__main_display_title_lbl, titleLine.c_str());

    // Details section
    lv_label_set_text(GUI_Label__cont__details_feels_like_value_lbl,
                      formatTemperature(data.feelsLike).c_str());
    lv_label_set_text(GUI_Label__cont__details_humidity_value_lbl,
                      formatHumidity(data.humidity).c_str());
    lv_label_set_text(GUI_Label__cont__details_wind_speed_lbl,
                      formatWind(data.windSpeed).c_str());

    // Hourly forecast
    lv_obj_t *const hourlyTimeLabels[4] = {
        GUI_Label__cont__hourly_forecast_item_1_time_lbl,
        GUI_Label__cont__hourly_forecast_item_2_time_lbl,
        GUI_Label__cont__hourly_forecast_item_3_time_lbl,
        GUI_Label__cont__hourly_forecast_item_4_time_lbl,
    };
    lv_obj_t *const hourlyTempLabels[4] = {
        GUI_Label__cont__hourly_forecast_item_1_temp_lbl,
        GUI_Label__cont__hourly_forecast_item_2_temp_lbl,
        GUI_Label__cont__hourly_forecast_item_3_temp_lbl,
        GUI_Label__cont__hourly_forecast_item_4_temp_lbl,
    };

    for (int i = 0; i < 4; ++i)
    {
        const HourItem &item = data.hourly[i];
        lv_label_set_text(hourlyTimeLabels[i], formatHour(item.ts, data.timezoneOffset).c_str());
        lv_label_set_text(hourlyTempLabels[i], formatTemperature(item.temp).c_str());
    }

    // Daily forecast
    lv_obj_t *const dailyDayLabels[3] = {
        GUI_Label__cont__daily_forecast_monday_item_day_lbl,
        GUI_Label__cont__daily_forecast_tuesday_item_day_lbl,
        GUI_Label__cont__daily_forecast_wednesday_item_day_lbl,
    };
    lv_obj_t *const dailyTempLabels[3] = {
        GUI_Label__cont__daily_forecast_monday_item_temp_lbl,
        GUI_Label__cont__daily_forecast_tuesday_item_temp_lbl,
        GUI_Label__cont__daily_forecast_wednesday_item_temp_lbl,
    };
    lv_obj_t *const dailyStatusLabels[3] = {
        GUI_Label__cont__daily_forecast_monday_item_status_lbl,
        GUI_Label__cont__daily_forecast_tuesday_item_status_lbl,
        GUI_Label__cont__daily_forecast_wednesday_item_status_lbl,
    };

    for (int i = 0; i < 3; ++i)
    {
        const DayItem &item = data.daily[i];
        lv_label_set_text(dailyDayLabels[i], formatWeekday(item.ts, data.timezoneOffset).c_str());
        lv_label_set_text(dailyTempLabels[i], formatDailyTemp(item).c_str());
        lv_label_set_text(dailyStatusLabels[i], fallbackText(item.main).c_str());
    }

    // AQI & indoor stats (using outdoor data as placeholder if indoor not available)
    const int aqiPercent = aqiPercentFromLevel(data.aqi);
    if (aqiPercent > 0)
    {
        String aqiPercentText = String(aqiPercent) + "%";
        lv_label_set_text(GUI_Label__cont__indoor_stats_air_quality_value_lbl, aqiPercentText.c_str());
    }
    else
    {
        lv_label_set_text(GUI_Label__cont__indoor_stats_air_quality_value_lbl, "--");
    }
    lv_slider_set_value(GUI_Slider__cont__indoor_stats_air_quality_slider, aqiPercent, LV_ANIM_OFF);

    lv_label_set_text(GUI_Label__cont__indoor_stats_air_quality_gap_value_lbl,
                      fallbackText(data.aqiLabel, "--").c_str());

    lv_label_set_text(GUI_Label__cont__indoor_stats_temp_value_lbl,
                      isnan(data.indoorTemp) ? "--" : formatTemperature(data.indoorTemp).c_str());
    lv_label_set_text(GUI_Label__cont__indoor_stats_humidity_value_lbl,
                      data.indoorHumidity >= 0 ? formatHumidity(data.indoorHumidity).c_str() : "--");

    // Alert message
    if (data.hasAlert)
    {
        lv_label_set_text(GUI_Label__cont__alert_message_lbl,
                          fallbackText(data.alert.description, "Weather alert").c_str());
    }
    else
    {
        lv_label_set_text(GUI_Label__cont__alert_message_lbl, "No severe alerts.");
    }
}
