#include <TFT_eSPI.h>
#include <lvgl.h>
#include <math.h>

#include "Adafruit_SHT4x.h"

#include "e1001_display.h"
#include "ui/GUI.h"

#include <WiFi.h>

#include "weather.h"
#include "weather_view.h"

e1001_driver_t e1001_driver;
Adafruit_SHT4x sht4 = Adafruit_SHT4x();
// ==================== Configuration ====================
const char *WIFI_SSID = "YOUR_WIFI_SSID";
const char *WIFI_PASSWORD = "YOUR_WIFI_PASSWORD";
const char *OW_API_KEY = "YOUR_API_KEY";    // OpenWeather API Key

const char *UNITS = "metric";
const char *LANG = "en";

const uint32_t WEATHER_REFRESH_INTERVAL_MIN = 60;  // Weather API call interval
const uint32_t INDOOR_REFRESH_INTERVAL_MIN = 10;   // Indoor sensor/UI refresh interval
const uint32_t WEATHER_REFRESH_INTERVAL_MS = WEATHER_REFRESH_INTERVAL_MIN * 60UL * 1000UL;
const uint32_t INDOOR_REFRESH_INTERVAL_MS = INDOOR_REFRESH_INTERVAL_MIN * 60UL * 1000UL;

// Mode 1: City name (recommended to include country code to avoid ambiguity).
String CITY = "Shenzhen,CN";

// Mode 2: Latitude / longitude
const char *LAT = "40.7128";
const char *LON = "-74.0060";

// Switch: true = city mode, false = coordinate mode.
bool USE_CITY_MODE = true;

typedef struct {
    float    temperature;
    float    humidity;
} SHT40_DATA;

WeatherRequest g_weatherRequest;
WeatherData g_weatherData;
bool g_weatherValid = false;
uint32_t g_lastWeatherFetchMs = 0;
uint32_t g_lastIndoorUpdateMs = 0;


// Reads temperature/humidity from the SHT40 sensor.
SHT40_DATA sht40_read()
{
    sensors_event_t humidity, temp;
    sht4.getEvent(&humidity, &temp);
    SHT40_DATA d;
    d.temperature   = temp.temperature;
    d.humidity      = humidity.relative_humidity;
    return d;
}

// Dedicated LVGL task loop.
void _lvgl_task(void *p)
{
    while(1)
    {
        lv_timer_handler();
        delay(50);
    }
}

// Initializes peripherals, WiFi, weather state, and the UI.
void setup()
{
    Serial1.begin(115200, SERIAL_8N1, 44, 43);
    e1001_display_init(&e1001_driver);
    GUI_init();
    Wire.setPins(19, 20);
    sht4.begin();

    WiFi.begin(WIFI_SSID, WIFI_PASSWORD);
    Serial1.print("Connecting to WiFi...");
    while (WiFi.status() != WL_CONNECTED)
    {
        delay(500);
        Serial1.print(".");
    }
    Serial1.println(" Connected!");
    Serial1.println("IP address: " + WiFi.localIP().toString());

    g_weatherRequest.apiKey = OW_API_KEY;
    g_weatherRequest.units = UNITS;
    g_weatherRequest.lang = LANG;
    g_weatherRequest.useCity = USE_CITY_MODE;
    if (USE_CITY_MODE)
    {
        g_weatherRequest.city = CITY;
    }
    else
    {
        g_weatherRequest.lat = LAT;
        g_weatherRequest.lon = LON;
    }

    if (fetchWeather(g_weatherRequest, g_weatherData))
    {
        SHT40_DATA indoor = sht40_read();
        g_weatherData.indoorTemp = indoor.temperature;
        g_weatherData.indoorHumidity = static_cast<int>(roundf(indoor.humidity));

        GUI_update_weather_screen(g_weatherData);
        g_weatherValid = true;
        g_lastWeatherFetchMs = millis();
        g_lastIndoorUpdateMs = g_lastWeatherFetchMs;
    }
    else
    {
        Serial1.println("Fetch weather failed.");
    }

    xTaskCreatePinnedToCore(_lvgl_task,"_lvgl_task",10 * 1024,NULL,1,NULL,1);
}


// Periodically refreshes weather data and updates the UI/display.
void loop()
{
    const uint32_t now = millis();

    if (!g_weatherValid || (now - g_lastWeatherFetchMs) >= WEATHER_REFRESH_INTERVAL_MS)
    {
        WeatherData latest;
        if (fetchWeather(g_weatherRequest, latest))
        {
            // Preserve latest indoor readings
            latest.indoorTemp = g_weatherData.indoorTemp;
            latest.indoorHumidity = g_weatherData.indoorHumidity;

            g_weatherData = latest;
            g_weatherValid = true;
            g_lastWeatherFetchMs = now;
            GUI_update_weather_screen(g_weatherData);
        }
    }


    if ((now - g_lastIndoorUpdateMs) >= INDOOR_REFRESH_INTERVAL_MS)
    {
        SHT40_DATA indoor = sht40_read();
        if (g_weatherValid)
        {
            g_weatherData.indoorTemp = indoor.temperature;
            g_weatherData.indoorHumidity = static_cast<int>(roundf(indoor.humidity));
            GUI_update_weather_screen(g_weatherData);
        }
        g_lastIndoorUpdateMs = now;
    }

    
    // Check if display refresh is needed
    if (e1001_display_should_refresh(&e1001_driver))
    {
        Serial1.println("Refreshing e-paper display...");
        Serial1.flush();
        e1001_display_refresh(&e1001_driver);
        Serial1.println("Display refresh complete");
        Serial1.flush();
    }
}
