#include "weather_view.h"

#include "ui/GUI.h"

#include <lvgl.h>
#include <algorithm>
#include <math.h>
#include <time.h>
#include <Arduino.h> // For Serial1 and ESP heap helpers

namespace
{
constexpr const char *kTempUnit = "°C";

String formatTemperature(float value)
{
    if (isnan(value))
        return "--";
    int rounded = static_cast<int>(roundf(value));
    return String(rounded) + "°";
}

String formatTemperatureNumber(float value)
{
    if (isnan(value))
        return "--";
    int rounded = static_cast<int>(roundf(value));
    return String(rounded);
}

String formatTemperatureWithUnit(float value)
{
    if (isnan(value))
        return String("--") + kTempUnit;
    int rounded = static_cast<int>(roundf(value));
    return String(rounded) + kTempUnit;
}

String formatHumidity(int value)
{
    if (value < 0)
        return "--";
    return String(value) + "%";
}

String formatWindKmH(float speedMs)
{
    if (isnan(speedMs))
        return "--";
    float kmh = speedMs * 3.6f;
    int rounded = static_cast<int>(roundf(kmh));
    return String(rounded);
}

String formatHour(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    char buf[6];
    snprintf(buf, sizeof(buf), "%02d:%02d", tmv.tm_hour, tmv.tm_min);
    return String(buf);
}

String formatWeekdayShort(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    static const char *kWeekdays[] = {"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"};
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    int idx = tmv.tm_wday;
    if (idx < 0 || idx > 6)
        return "--";
    return String(kWeekdays[idx]);
}

String formatDateLine(uint32_t ts, int tzOffset)
{
    if (ts == 0)
        return "--";
    time_t t = static_cast<time_t>(ts) + tzOffset;
    struct tm tmv;
    gmtime_r(&t, &tmv);
    char buf[16];
    snprintf(buf, sizeof(buf), "%02d.%02d.%04d", tmv.tm_mday, tmv.tm_mon + 1, tmv.tm_year + 1900);
    return String(buf);
}

String formatDailyTemp(const DayItem &day)
{
    if (isnan(day.min) && isnan(day.max) && isnan(day.day))
        return "--";

    float base = !isnan(day.day) ? day.day : day.max;
    if (isnan(base))
        base = day.min;
    if (isnan(base))
        return "--";
    return formatTemperature(base);
}

String fallbackText(const String &value, const char *fallback = "--")
{
    return value.length() > 0 ? value : String(fallback);
}

String formatCurrentStatus(const WeatherData &data)
{
    const String primary = fallbackText(data.desc, "");
    if (primary.length() > 0)
        return primary;
    return fallbackText(data.main, "Loading...");
}

int aqiPercentFromLevel(int aqi)
{
    switch (aqi)
    {
    case 1: return 100;
    case 2: return 80;
    case 3: return 60;
    case 4: return 40;
    case 5: return 20;
    default: return 0;
    }
}

} // namespace

// Safety helpers: if a widget pointer is null, print an error and skip the update.
#define SAFE_LABEL(obj, val) \
    do { \
        if ((obj) == NULL) { \
            Serial1.printf("ERROR: %s is NULL (skip label update)\n", #obj); \
        } else { \
            String txt = (val); \
            /* Serial1.printf("OK: %s -> %s\n", #obj, txt.c_str()); */ \
            lv_label_set_text(obj, txt.c_str()); \
        } \
    } while(0)

#define SAFE_SLIDER(obj, val, range_max) \
    do { \
        if ((obj) == NULL) { \
            Serial1.printf("ERROR: %s is NULL (skip slider update)\n", #obj); \
        } else { \
            lv_slider_set_range(obj, 0, range_max); \
            lv_slider_set_value(obj, val, LV_ANIM_OFF); \
        } \
    } while(0)

#define SAFE_IMG_FLAG(obj, flag, set) \
    do { \
        if ((obj) == NULL) { \
             Serial1.printf("ERROR: %s is NULL (skip image flag update)\n", #obj); \
        } else { \
            if (set) lv_obj_add_flag(obj, flag); \
            else lv_obj_clear_flag(obj, flag); \
        } \
    } while(0)


// Updates LVGL widgets with the latest weather data.
void GUI_update_weather_screen(const WeatherData &data)
{
    // 1. Forecast panel: current status
    SAFE_LABEL(GUI_Label__main_screen__current_status_lbl, formatCurrentStatus(data));

    // 2. Forecast panel: next 5 days
    lv_obj_t *const day5DayLabels[5] = {
        GUI_Label__main_screen__5day_item_1_day_lbl,
        GUI_Label__main_screen__5day_item_2_day_lbl,
        GUI_Label__main_screen__5day_item_3_day_lbl,
        GUI_Label__main_screen__5day_item_4_day_lbl,
        GUI_Label__main_screen__5day_item_5_day_lbl,
    };
    lv_obj_t *const day5TempLabels[5] = {
        GUI_Label__main_screen__5day_item_1_temp_lbl,
        GUI_Label__main_screen__5day_item_2_temp_lbl,
        GUI_Label__main_screen__5day_item_3_temp_lbl,
        GUI_Label__main_screen__5day_item_4_temp_lbl,
        GUI_Label__main_screen__5day_item_5_temp_lbl,
    };

    for (int i = 0; i < 5; ++i)
    {
        if (i < weather_day_count)
        {
            const DayItem &item = data.daily[i];
            SAFE_LABEL(day5DayLabels[i], formatWeekdayShort(item.ts, data.timezoneOffset));
            SAFE_LABEL(day5TempLabels[i], formatDailyTemp(item));
        }
        else
        {
            SAFE_LABEL(day5DayLabels[i], "--");
            SAFE_LABEL(day5TempLabels[i], "--");
        }
    }

    // 3. Forecast panel: hourly
    lv_obj_t *const hourlyTimeLabels[3] = {
        GUI_Label__main_screen__daily_item_1_time_lbl,
        GUI_Label__main_screen__daily_item_2_time_lbl,
        GUI_Label__main_screen__daily_item_3_time_lbl,
    };
    lv_obj_t *const hourlyTempLabels[3] = {
        GUI_Label__main_screen__daily_item_1_temp_lbl,
        GUI_Label__main_screen__daily_item_2_temp_lbl,
        GUI_Label__main_screen__daily_item_3_temp_lbl,
    };
    lv_obj_t *const hourlyHumidityLabels[3] = {
        GUI_Label__main_screen__daily_item_1_humidity_lbl,
        GUI_Label__main_screen__daily_item_2_humidity_lbl,
        GUI_Label__main_screen__daily_item_3_humidity_lbl,
    };

    const String currentHumidity = formatHumidity(data.humidity);
    for (int i = 0; i < 3; ++i)
    {
        const HourItem &item = data.hourly[i];
        SAFE_LABEL(hourlyTimeLabels[i], formatHour(item.ts, data.timezoneOffset));
        SAFE_LABEL(hourlyTempLabels[i], formatTemperature(item.temp));
        SAFE_LABEL(hourlyHumidityLabels[i], currentHumidity);
    }

    // 4. Dashboard: location and headline
    SAFE_LABEL(GUI_Label__main_screen__location_city_lbl, fallbackText(data.cityName, "--"));
    
    String dateText = formatDateLine(data.dt, data.timezoneOffset);
    if (dateText != "--") dateText += ".";
    SAFE_LABEL(GUI_Label__main_screen__location_date_lbl, dateText);
    
    SAFE_LABEL(GUI_Label__main_screen__main_temp_value_lbl, formatTemperatureNumber(data.temp));
    SAFE_LABEL(GUI_Label__main_screen__main_temp_unit_lbl, kTempUnit);

    // 5. Dashboard: details
    SAFE_LABEL(GUI_Label__main_screen__details_feels_like_value_lbl, formatTemperature(data.feelsLike));
    SAFE_LABEL(GUI_Label__main_screen__details_humidity_value_lbl, formatHumidity(data.humidity));
    SAFE_LABEL(GUI_Label__main_screen__details_wind_speed_lbl, formatWindKmH(data.windSpeed));
    SAFE_LABEL(GUI_Label__main_screen__details_wind_unit_lbl, "km/h");

    // 6. Dashboard: indoor
    SAFE_LABEL(GUI_Label__main_screen__indoor_temp_value_lbl, formatTemperatureWithUnit(data.indoorTemp));
    SAFE_LABEL(GUI_Label__main_screen__indoor_humidity_value_lbl, formatHumidity(data.indoorHumidity));

    // 7. Dashboard: air quality
    const int aqiPercent = aqiPercentFromLevel(data.aqi);
    SAFE_SLIDER(GUI_Slider__main_screen__air_quality_slider, aqiPercent, 100);
    SAFE_LABEL(GUI_Label__main_screen__air_quality_status_lbl, fallbackText(data.aqiLabel, "--"));

    // 8. Alerts
    if (data.hasAlert)
    {
        SAFE_IMG_FLAG(GUI_Image__main_screen__alert_warning_icon_img, LV_OBJ_FLAG_HIDDEN, false); // Show
        SAFE_LABEL(GUI_Label__main_screen__alert_title_lbl, fallbackText(data.alert.event, "Weather alert"));
        SAFE_LABEL(GUI_Label__main_screen__alert_description_lbl, fallbackText(data.alert.description, "Details unavailable"));
    }
    else
    {
        SAFE_IMG_FLAG(GUI_Image__main_screen__alert_warning_icon_img, LV_OBJ_FLAG_HIDDEN, true); // Hide
        SAFE_LABEL(GUI_Label__main_screen__alert_title_lbl, "No alerts");
        SAFE_LABEL(GUI_Label__main_screen__alert_description_lbl, "");
    }
}
